<?php declare(strict_types = 1);

namespace Blog\FrontModule\Model;

use Blog\Model\Entities\Article;
use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Images\ImagePipe;
use Doctrine\ORM\Query;
use Gallery\Model\Images;
use IPub\MobileDetect\MobileDetect;
use Kdyby\Doctrine\QueryBuilder;
use Kdyby\Translation\Translator;
use Nette\Utils\DateTime;

/**
 * Class Articles
 * @package Blog\FrontModule\Model
 */
class Articles extends BaseFrontEntityService
{
	/** @var \stdClass */
	protected $entityClass = Article::class;

	/** @var Translator */
	public $translator;

	/** @var ImagePipe */
	protected $imagePipe;

	/** @var Images */
	protected $imagesService;

	/** @var MobileDetect */
	protected $mobileDetect;

	const CACHE_NAMESPACE = 'articles';

	public function __construct(Translator $translator, ImagePipe $imagePipe, Images $images, MobileDetect $mobileDetect)
	{
		$this->translator    = $translator;
		$this->imagePipe     = $imagePipe;
		$this->imagesService = $images;
		$this->mobileDetect  = $mobileDetect;
	}

	/**
	 * Kritéria pro publikované články
	 *
	 * @param QueryBuilder $qb
	 * @param bool         $withLockedCategory
	 *
	 * @return QueryBuilder $qb
	 */
	public function setPublishedCriteria(QueryBuilder $qb, $withLockedCategory = false)
	{
		$qb->andWhere('a.publishUp < :now')->andWhere('a.publishDown IS NULL OR a.publishDown < :now')
			->andWhere('a.title IS NOT NULL')->andWhere('a.isPublished = 1')
			->andWhere('c.isPublished = 1')->andWhere('c.lang IS NULL OR c.lang = :locale')
			->setParameter('now', new DateTime())->setParameter('locale', $this->translator->getLocale());

		if (!$withLockedCategory) {
			$qb->andWhere('c.password IS NULL');
		}

		return $qb;
	}

	/**
	 * Získání id a aliasu pro navigaci
	 *
	 * @param $id
	 *
	 * @return array|mixed
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function getForNavigation($id)
	{
		$qb = $this->getEr()->createQueryBuilder('a')->select('a.id, a.alias')
			->andWhere(is_array($id) ? 'a.id IN (:id)' : 'a.id = :id')->setParameter('id', $id)
			->getQuery();

		return is_array($id) ? $qb->getArrayResult() : $qb->getOneOrNullResult(Query::HYDRATE_ARRAY);
	}

	/**
	 * @param      $id
	 * @param bool $withLockedCategory
	 *
	 * @return mixed
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function get($id, $withLockedCategory = false)
	{
		$qb = $this->getEr()->createQueryBuilder('a');
		$qb = $this->setPublishedCriteria($qb, $withLockedCategory);

		$qb->join('a.category', 'c')->addSelect('c')->andWhere('a.id = :id')->setParameter('id', $id)
			->leftJoin('a.gallery', 'g')->join('a.createdBy', 'cb')->addSelect('g, cb');

		$article = $qb->getQuery()->getOneOrNullResult();

		if ($article && $article->gallery)
			$this->imagesService->getEr()->createQueryBuilder('i')->andWhere('i.isPublished = 1')
				->andWhere('i.album = :album')->setParameter('album', $article->gallery->getId());

		return $article;
	}

	/**
	 * Získání všech zveřejněných článků
	 *
	 * @param int|array    $categories
	 * @param int|null     $offset
	 * @param int|null     $limit
	 * @param bool         $withLockedCategories
	 * @param string|array $sort
	 *
	 * @return mixed
	 */
	public function getAll($categories = null, $offset = null, $limit = null, $sort = 'publishUp', $withLockedCategories = false)
	{
		$qb = $this->getEr()->createQueryBuilder('a');
		$qb = $this->setPublishedCriteria($qb, $withLockedCategories);
		$qb->join('a.category', 'c')->leftJoin('a.gallery', 'g')->leftJoin('g.images', 'i')->addSelect('c, g');

		// Kategorie
		if (is_array($categories) && count($categories) == 1)
			$categories = $categories[0];
		if ($categories) {
			$qb->andWhere(is_numeric($categories) ? 'c.id = :catId' : 'c.id IN (:catId)')
				->setParameter('catId', $categories);
		}

		// Offset a limit
		if ($offset)
			$qb->setFirstResult($offset);
		if ($limit)
			$qb->setMaxResults($limit);

		// Pořadí
		if (is_string($sort)) {
			switch ($sort) {
				case 'default':
				case 'publishUp':
					$qb->orderBy('a.publishUp', 'DESC');
					break;
			};
		} else if (is_array($sort)) {
			foreach ($sort as $k => $v) {
				$qb->addOrderBy($k, $v);
			}
		}

		$qb->groupBy('a.id');

		return $qb->getQuery()->getResult();
	}

	/**
	 * @param null $categories
	 * @param bool $withLockedCategories
	 *
	 * @return mixed
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function getAllCount($categories = null, $withLockedCategories = false)
	{
		$qb = $this->getEr()->createQueryBuilder('a');
		$qb = $this->setPublishedCriteria($qb, $withLockedCategories);
		$qb->select('count(a.id)')->join('a.category', 'c');

		// Kategorie
		if (is_array($categories) && count($categories) == 1)
			$categories = $categories[0];
		if ($categories) {
			$qb->andWhere(is_numeric($categories) ? 'c.id = :catId' : 'c.id IN (:catId)')
				->setParameter('catId', $categories);
		}

		return $qb->getQuery()->getSingleScalarResult();
	}

	/**
	 * @param Article        $article
	 * @param int|array|null $categories
	 * @param                $limit
	 * @param bool           $withLockedCategories
	 *
	 * @return Article[]|null
	 */
	public function getNextPublishedArticle(Article $article, $categories = null, $limit = 1, $withLockedCategories = false)
	{
		$qb = $this->getEr()->createQueryBuilder('a');
		$qb = $this->setPublishedCriteria($qb, $withLockedCategories);

		$qb->join('a.category', 'c')
			->andWhere('a.publishUp > :articlePublishUp')->setParameter('articlePublishUp', $article->publishUp)
			->orderBy('a.publishUp', 'ASC');

		if (is_array($categories) && count($categories) == 1)
			$categories = $categories[0];
		if ($categories)
			$qb->andWhere(is_array($categories) ? 'c.id IN (:cat)' : 'c.id = :cat')->setParameter('cat', $categories);

		return $qb->setMaxResults($limit)->getQuery()->getResult();
	}

	/**
	 * @param Article        $article
	 * @param int|array|null $categories
	 * @param                $limit
	 * @param bool           $withLockedCategories
	 *
	 * @return Article[]|null
	 */
	public function getPrevPublishedArticle(Article $article, $categories = null, $limit = 1, $withLockedCategories = false)
	{
		$qb = $this->getEr()->createQueryBuilder('a');
		$qb = $this->setPublishedCriteria($qb, $withLockedCategories);

		$qb->join('a.category', 'c')
			->andWhere('a.publishUp < :articlePublishUp')->setParameter('articlePublishUp', $article->publishUp)
			->orderBy('a.publishUp', 'DESC');

		if (is_array($categories) && count($categories) == 1)
			$categories = $categories[0];
		if ($categories)
			$qb->andWhere(is_array($categories) ? 'c.id IN (:cat)' : 'c.id = :cat')->setParameter('cat', $categories);

		return $qb->setMaxResults($limit)->getQuery()->getResult();
	}

	/**
	 * Zpracování fulltextu (když ckeditor obsahuje vychytávky)
	 *
	 * @param Article $article
	 *
	 * @return string
	 */
	public function processFulltext($article)
	{
		return $article->getFulltext();
	}

	/**
	 * Přidání zobrazení ke článku
	 *
	 * @param int $articleId
	 */
	public function addShow(int $articleId)
	{
		$this->getEr()->createQuery('UPDATE ' . $this->entityClass . ' a SET a.shows = a.shows + 1 WHERE a.id = :id')
			->setParameters([':id' => $articleId])->execute();
	}
}