<?php declare(strict_types = 1);

namespace Blog\Model;

use Blog\Model\Entities\Article;
use Blog\Model\Entities\Category;
use Blog\Model\Entities\Hit;
use Blog\Model\Entities\Hit2;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Helpers\BaseService;
use Core\Model\Helpers\Traits\EntityManagerTrait;
use Core\Model\Helpers\Traits\TranslatorTrait;
use Core\Model\Images\Image;
use Core\Model\Images\ImagePipe;
use Core\Model\Lang\Langs;
use Doctrine\Common\Collections\Criteria;
use Gallery\Model\Images;
use IPub\MobileDetect\MobileDetect;
use Kdyby\Doctrine\EntityRepository;
use Kdyby\Doctrine\QueryBuilder;
use Kdyby\Translation\Translator;
use Nette\Caching\Cache;
use Nette\Utils\DateTime;
use Nette\Utils\Html;
use Tags\Model\Entities\Tag;
use function SimpleHtmlDom\str_get_html;

/**
 * Class Articles
 * @package Blog\Model
 *
 * @method Article|null|object = getReference($id)
 * @method Article[]|null getAll()
 * @method Article|null get($id)
 */
class Articles extends BaseEntityService
{
	protected $entityClass = Article::class;

	/** @var Translator @inject */
	public $translator;

	/** @var int */
	private static $i;

	/** @var ImagePipe */
	protected $imagePipe;

	/** @var Images */
	protected $imagesService;

	/** @var MobileDetect */
	protected $mobileDetect;

	const CACHE_NAMESPACE = 'articles';

	public function __construct(ImagePipe $imagePipe, Images $images, MobileDetect $mobileDetect)
	{
		$this->imagePipe     = $imagePipe;
		$this->imagesService = $images;
		$this->mobileDetect  = $mobileDetect;
	}

	private function getCache()
	{
		if ($this->cache === null)
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);

		return $this->cache;
	}

	/**
	 * Return published articles by ID(s)
	 *
	 * @param int|array $id
	 *
	 * @return Article[]|Article|null
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function getPublishedById($id)
	{
		$articles = $this->getPublishedArticlesQuery()
			->andWhere(is_array($id) ? 'a.id IN (:aId)' : 'a.id = :aId')->setParameter('aId', $id)
			->getQuery();

		if (is_numeric($id))
			return $articles->getOneOrNullResult();

		return $articles->getResult();
	}

	/**
	 * @param Article $article
	 * @param         $limit
	 *
	 * @return Article[]|null
	 */
	public function getNextPublishedArticle($article, $limit = 1)
	{
		return $this->getPublishedArticlesQuery()
			->andWhere('a.publishUp > :articlePublishUp')->setParameter('articlePublishUp', $article->publishUp)
			->orderBy('a.publishUp', 'ASC')
			->setMaxResults($limit)->getQuery()->useResultCache(true, 120)->getResult();
	}

	/**
	 * @param Article $article
	 * @param         $limit
	 *
	 * @return Article[]|null
	 */
	public function getPrevPublishedArticle($article, $limit = 1)
	{
		return $this->getPublishedArticlesQuery()
			->andWhere('a.publishUp < :articlePublishUp')->setParameter('articlePublishUp', $article->publishUp)
			->orderBy('a.publishUp', 'DESC')
			->setMaxResults($limit)->getQuery()->useResultCache(true, 120)->getResult();
	}

	/**
	 * @param $id
	 *
	 * @return null|Article
	 */
	public function getArticle($id)
	{
		return $this->getEr()->find($id);

		$key = self::CACHE_NAMESPACE . '/article/' . $this->translator->getLocale() . '-' . $id;

		return $this->getCache()->load($key, function(&$dep) use ($key, $id) {
			$dep = [Cache::TAGS => [$key, self::CACHE_NAMESPACE, Articles::CACHE_NAMESPACE . '/article/' . $id]];

			/** @var Article $article */
			$article = $this->getEr()->find($id);
			$article->getImage();

			return $article;
		});
	}

	/**
	 * @return Article|null
	 */
	public function getFeatured()
	{
		$key = self::CACHE_NAMESPACE . '/featured/' . $this->translator->getLocale();

		return $this->getCache()->load($key, function(&$dep) use ($key) {
			$dep = [Cache::TAGS   => [$key, self::CACHE_NAMESPACE, self::CACHE_NAMESPACE . '/featured'],
			        Cache::EXPIRE => '5 minutes'];

			/** @var Article $article */
			$article = $this->getPublishedArticlesQuery()->andWhere('a.featured IS NOT null')->setMaxResults(1)
				->getQuery()->getOneOrNullResult();

			if (!$article)
				return null;

			$article->getImage();
			$article->getCategory()->title;

			return $article;
		});
	}

	/**
	 * @return Article[]|null
	 */
	public function getAvailableArticles()
	{
		return $this->getEr()->findAll();
	}

	public function publishedArticlesCriteria()
	{
		$expr     = Criteria::expr();
		$now      = new DateTime();
		$locale   = $this->translator->getLocale();
		$criteria = Criteria::create()
			->andWhere(
				$expr->andX(
					$expr->andX(
						$expr->orX($expr->isNull('a.publishUp'), $expr->lte('a.publishUp', $now)),
						$expr->orX($expr->isNull('a.publishDown'), $expr->gte('a.publishDown', $now))
					),
					$expr->neq('a.publishUp', null),
					$expr->neq('a.title', null),
					$expr->neq('a.category', null),
					$expr->eq('a.isPublished', 1),
					$expr->eq('c.isPublished', 1),
					$expr->isNull('c.password'),
					$expr->orX(
						$expr->isNull('c.lang'),
						$expr->eq('c.lang', $locale),
						$expr->eq('c.lang', '')
					)
				)
			);

		return $criteria;
	}

	/**
	 * @param QueryBuilder|\Doctrine\ORM\QueryBuilder $qb
	 *
	 * @throws \Doctrine\ORM\Query\QueryException
	 */
	public function publishedArticlesSetQB(&$qb)
	{
		$qb->leftJoin(Category::class, 'c', 'WITH', 'a.category = c.id')
			->addCriteria($this->publishedArticlesCriteria());
	}

	/**
	 * @return \Doctrine\ORM\QueryBuilder
	 * TODO udělat přes funkci publishedArticlesCriteria
	 */
	public function getPublishedArticlesQuery($withLocked = false, $skip = [])
	{
		$qb = $this->getEr()->createQueryBuilder('a')
			//			->innerJoin('a.tags', 'tags')
			->andWhere('a.title IS NOT null')->andWhere('a.category IS NOT null')
			->andWhere('a.isPublished = 1')
			//			->andWhere('(a.lang IS null OR a.lang = :locale OR a.lang = \'\')') // Jazyk podle kateogrie
			->leftJoin(Category::class, 'c', 'WITH', 'a.category = c.id')
			->andWhere('c.isPublished = 1')
			->andWhere('(c.lang IS null OR c.lang = :locale OR c.lang = \'\')')
			->setParameter('now', new DateTime())
			->setParameter('locale', $this->translator->getLocale())
			->groupBy('a.id');

		$publish = ['(a.publishDown >= :now OR a.publishDown IS null)'];
		if (!in_array('publishUp', $skip)) {
			$publish[] = '(a.publishUp IS null OR a.publishUp <= :now)';
			$publish[] = 'a.publishUp IS NOT null';
		}

		$qb->andWhere(implode(' AND ', $publish));

		if (!$withLocked)
			$qb->andWhere('c.password IS null');

		return $qb;
	}

	public function getPublishedArticles()
	{
		return $this->getPublishedArticlesQuery()->getQuery()->getResult();
	}

	public function removeArticle($articleId)
	{
		if ($article = $this->getEr()->find($articleId)) {
			$this->em->remove($article);
			$this->em->flush();

			return true;
		}

		return false;
	}

	public function removeArticles($ids)
	{
		$errors = [];
		foreach ($ids as $id) {
			if (!$this->removeArticle($id))
				$errors[] = $id;
		}

		return $errors;
	}

	/**
	 * @param int $articleId
	 * @param int $featureTime
	 *
	 * @return bool
	 * @throws \Exception
	 */
	public function setFeature($articleId, $featureTime)
	{
		/** @var Article $article */
		if ($article = $this->getEr()->find($articleId)) {
			if ($featureTime > 1) {
				$endTime = (new DateTime())->modify("+$featureTime minutes");
			} else {
				$endTime = $featureTime;
			}

			$this->em->createQueryBuilder()->update(Article::class, 'a')->set('a.featured', 'null')
				->where('a.featured IS NOT null')
				->andWhere('a.category IN (SELECT c.id FROM ' . Category::class . ' c WHERE c.lang = :lang)')// jazyk podle kategorie
				->setParameter('lang', $article->category->getLangId())
				->getQuery()->execute();

			$article->setFeatured((string) $endTime);
			$this->em->persist($article);
			$this->em->flush();
			$this->getCache()->clean([Cache::TAGS => [Articles::CACHE_NAMESPACE . '/list',
				Articles::CACHE_NAMESPACE . '/featured']]);

			return true;
		}

		return false;
	}

	public function setPublish($id, $state)
	{
		if ($item = $this->getReference($id)) {
			$item->isPublished = $state;
			$this->em->persist($item);
			$this->em->flush();

			return true;
		}

		return false;
	}

	public function checkFeatured()
	{
		$langs            = array_keys(Langs::$langs);
		$featuredArticles = [];

		foreach ($langs as $v)
			$featuredArticles[$v] = null;
		foreach ($this->getEr()->findBy(['featured !=' => null]) as $article) {
			/** @var Article $article */
			$featuredArticles[$article->getCategory()->getLangId()] = $article;
		}

		try {
			foreach ($featuredArticles as $lang => $featured) {
				if ($featured == null
					|| ($featured->featured != 1 && (new DateTime('now'))->getTimestamp() > DateTime::from($featured->featured)->getTimestamp())
					|| ($featured->featured == 1)
				) {
					$lastArticle = $this->getPublishedArticlesQuery()->setParameter('locale', $lang)->orderBy('a.publishUp', 'DESC')->setMaxResults(1)->getQuery()->getSingleResult();
					if (!$featured || $lastArticle->getId() != $featured->getId()) {
						$this->setFeature($lastArticle->getId(), 1);
					}
				}
			}
		} catch (\Exception $e) {
		}
	}

	public function getArticlesCountInCategoryByPublishing($categoryId)
	{
		$arr = [
			0 => ['count' => 0, 'isPublished' => 0],
			1 => ['count' => 0, 'isPublished' => 1],
		];
		$qb  = $this->getEr()->createQueryBuilder('a', 'a.isPublished')->select('COUNT(a.id) as count, a.isPublished')
			->where('a.category = :category')->setParameter('category', $categoryId)
			->groupBy('a.isPublished');

		return (($qb->getQuery()->getResult() ?: []) + $arr);
	}

	/**
	 * @param $articleId
	 *
	 * @return int[]|null
	 */
	public function getRelated($articleId, $limit = 1)
	{
		try {
			$key = self::CACHE_NAMESPACE . '/related/' . $articleId;

			$ids = $this->getCache()->load($key, function(&$dep) use ($key, $articleId) {
				$dep     = [Cache::TAGS => [$key, self::CACHE_NAMESPACE], Cache::EXPIRE => '1 day'];
				$article = $this->getReference($articleId);

				$qb = $this->getPublishedArticlesQuery();

				$ids    = [];
				$result = $qb->select('a.id')
					->andWhere('a.publishUp > :from')
					->join(Tag::class, 't', 'WITH', 't.id IN (:tags)')
					->groupBy('a.id')
					->setParameter('tags', $article->getTagsId())
					->setParameter('from', DateTime::createFromFormat('Y-m-d', '2015-01-01'))
					->setMaxResults(200)->orderBy('a.publishUp', 'DESC')->getQuery()->getArrayResult();

				foreach ($result as $r)
					$ids[] = $r['id'];

				return $ids;
			});

			shuffle($ids);
			$ids = array_slice($ids, 0, $limit);

			return $ids;
		} catch (\Exception $e) {
		}

		return null;
	}

	public function getPreparedText(&$article, $link)
	{
		if ($article->fulltext) {
			$key     = Articles::CACHE_NAMESPACE . '/article/' . $article->getId() . '/preparedFulltext';
			$scripts = [];

			$html = str_get_html($article->fulltext);
			foreach ($html->find('.socialshare') as $v) {
				$src = (string) $v->{'data-src'};
				if (strpos($src, 'youtube.com') !== false) {
					preg_match("#(?<=v=)[a-zA-Z0-9-]+(?=&)|(?<=v\/)[^&\n]+(?=\?)|(?<=v=)[^&\n]+|(?<=youtu.be/)[^&\n]+#", $src, $matches);

					if (isset($matches[0]))
						$v->outertext = '<div class="iframe-wrap"><iframe src="https://www.youtube.com/embed/' . $matches[0] . '" class="youtube-iframe" frameborder="0"></iframe></div>';
					else
						$v->outertext = '';
				} else if (strpos($src, 'vimeo.com') !== false) {
					$src          = explode('/', $src);
					$v->outertext = '<div class="iframe-wrap"><iframe src="https://player.vimeo.com/video/' . end($src) . '" frameborder="0" webkitallowfullscreen mozallowfullscreen allowfullscreen></iframe></div>';
				} else if (strpos($src, '.gifv') !== false || strpos($src, '.webm') !== false) {
					$src          = str_replace('.gifv', '.webm', $src);
					$v->outertext = '<div class="text-align: center;"><video style="max-width: 100%" preload="auto" autoplay="autoplay" loop="loop"><source src="' . $src . '" type="video/webm"></video></div>';
				} else if (strpos($src, '.gif') !== false) {
					$v->outertext = "<div style='text-align: center'><img src='{$src}'></div>";
				} else if (strpos($src, 'instagram.com') !== false) {
					$scripts['instagram'] = "<script async defer='defer' src='//platform.instagram.com/en_US/embeds.js'></script>";
					$v->outertext         = "<blockquote class='instagram-media' data-instgrm-captioned data-instgrm-version='4'>"
						. "<a href='{$src}'></a>"
						. "</blockquote>";
				} else if (strpos($src, 'facebook.com') !== false) {
					$v->outertext = "<div class='fb-post' data-href='{$src}' data-width='500' data-show-text='true'></div>";
				} else if (strpos($src, 'twitter.com') !== false) {
					$scripts['twitter'] = "<script async src='https://platform.twitter.com/widgets.js' charset='utf-8'></script>";
					$v->outertext       = "<blockquote class='twitter-tweet' data-lang='cs'><a href='{$src}'></a></blockquote>";
				} else if (strpos($src, 'tumblr.com') !== false) {
					$scripts['tumblr'] = '<script async src="https://assets.tumblr.com/post.js"></script>';
					$v->outertext      = '<div class="tumblr-post" data-href="https://embed.tumblr.com/embed/post/0FTBaMwb9fSe1ArdSIeGBg/171646778196" data-did="4840232ac009c4f4135503e8c14006b4120200a4"></div>';
				} else if (strpos($src, 'giphy.com') !== false) {
					$arr          = explode('-', $src);
					$v->outertext = "<div class='iframe-wrap'><iframe src='https://giphy.com/embed/" . array_pop($arr) . "?video=0' frameBorder='0' class='giphy-embed' allowFullScreen></iframe></div>";
				} else if (strpos($src, 'imgur.com') !== false) {
					$arr  = explode('/', $src);
					$code = array_pop($arr);
					if (end($arr) == 'gallery')
						$code = 'a/' . $code;
					$scripts['imgur'] = "<script async src='//s.imgur.com/min/embed.js' charset='utf-8'></script>";
					$v->outertext     = "<div style='text-align: center;'><blockquote class='imgur-embed-pub' data-id='" . $code . "'></blockquote></div>";
				} else if (strpos($src, 'spotify.com') !== false) {
					$v->outertext = '<iframe src="https://open.spotify.com/embed?uri=' . $src . '" width="300" height="380" frameborder="0" allowtransparency="true" allow="encrypted-media"></iframe>';
				}
			}

			foreach ($html->find('img') as $img) {
				if (strpos($img->src, 'images') === 0)
					$img->src = '/' . $img->src;
			}

			$html = implode('', $scripts) . $html;

			$article->preparedFulltext = (string) $html;
		} else {
			$article->preparedFulltext = $article->fulltext;
		}

		try {
			$relatedKey = "/<div class=\"relatedarticle\">(.*?)<\/div>/";
			$text       = $article->getPreparedFulltext();
			$count      = preg_match_all($relatedKey, $text);

			if (empty($article->getTagsId()) || $count == 0) {
				$text = preg_replace($relatedKey, '', $text);
			} else {
				$relatedArticles = $this->getRelated($article->getId(), $count);

				self::$i = 0;
				$text    = preg_replace_callback($relatedKey, function($match) use ($relatedArticles, $link) {
					if (!isset($relatedArticles[self::$i]))
						return '';
					$articleId = $relatedArticles[self::$i];
					self::$i++;
					$keyA = self::CACHE_NAMESPACE . '/article/' . $articleId;
					$key  = self::CACHE_NAMESPACE . '/relatedInArticleText/' . $articleId;

					return $this->getCache()->load($key, function(&$dep) use ($key, $keyA, $articleId, $link) {
						$dep = [Cache::TAGS => [$key, self::CACHE_NAMESPACE, $keyA], Cache::EXPIRE => '1 day'];

						$a   = $this->getArticle($articleId);
						$img = $a->getImageFile() ? $this->imagePipe->request($a->getImageFile(), '171x110', 'fill') : '#';

						return Html::el('div class=related-article')
							->addHtml(Html::el('a', ['href' => $link(':Blog:Front:Articles:detail', ['id' => $a->getId()])])
								->addHtml(Html::el('span class=img')->addHtml(Html::el('img', ['src' => $img])))
								->addHtml(Html::el('span class=text')
									->addHtml(Html::el('span class=date')->setText($a->publishUp->format('j. n. Y')))
									->addHtml(Html::el('span class=t')->setText($a->title))));
					});
				}, $text);
			}

			$imageKey = "/<div class=\"albumimages\" data-src=\"(\d*)\">(.*?)<\/div>/";
			$text     = preg_replace_callback($imageKey, function($match) {
				/** @var Image $img */
				if (!isset($match[1]) || !($img = $this->imagesService->get($match[1])))
					return '';

				return Html::el('div class=album-image')
					->addHtml('<img src="' . $img->getFilePath() . '">')
					->addHtml(Html::el('em class=album-image-description')->setText($img->description));
			}, $text);

			$article->preparedFulltext = $text;
		} catch (\Exception $e) {
		}

		$article->preparedFulltext = str_replace('[timestamp]', time(), $article->preparedFulltext);

		return $article->preparedFulltext;
	}

	public function getMostRead($limit = 5)
	{
		$qb = $this->em->getRepository(Hit2::class)->createQueryBuilder('h')
			->select('IDENTITY(h.article) as articleId, SUM(h.shows) as hits')
			->join(Article::class, 'a', 'WITH', 'a.id = h.article')
			->andWhere('h.date >= :dateLimit')->setParameter('dateLimit', (new DateTime())->modify('-3 days'))
			->groupBy('h.article')
			->orderBy('hits', 'DESC');

		$this->publishedArticlesSetQB($qb);
		$qb->setMaxResults($limit);

		return $qb->getQuery()->getArrayResult();
	}

	public function searchFulltext($word, $offset = 0, $limit = 20)
	{
		return $this->getPublishedArticlesQuery()->andWhere('a.title LIKE :word OR a.fulltext LIKE :word')
			->setParameter('word', "%$word%")
			->setFirstResult($offset)->setMaxResults($limit)
			->orderBy('a.publishUp', 'DESC')
			->getQuery()->getArrayResult();
	}
}