<?php declare(strict_types = 1);

namespace Blog\Model\Navigation;

use Blog\FrontModule\Model\Articles;
use Blog\Model\Categories;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Navigations\Model\Entities\Navigation;
use Navigations\Model\Providers\INavigationItem;
use Nette\Utils\ArrayHash;

class Category extends BaseNavigation implements INavigationItem
{
	protected $title                  = 'category';
	protected $presenter              = 'Blog:Front:Categories';
	protected $action                 = 'category';
	protected $presenterArticleDetail = 'Blog:Front:Articles';
	protected $actionArticleDetail    = 'detail';

	/** @var Categories */
	protected $categoriesService;

	/** @var Articles */
	protected $articlesService;

	public function __construct(Categories $categories, Articles $articles)
	{
		$this->categoriesService = $categories;
		$this->articlesService   = $articles;
	}

	/**
	 * @return BaseContainer
	 */
	public function getFormContainer()
	{
		$container = new BaseContainer();

		$categories = [];
		foreach ($this->categoriesService->getAll() as $category)
			$categories[$category->getId()] = $category->title;

		$sort = [];
		foreach (['default', 'random'] as $v)
			$sort[$v] = $this->t('blog.navigationBuilder.sortTypes.' . $v);

		$container->addSelect('category', 'blog.navigationBuilder.category', $categories);
		$container->addSelect('showPublishedInFuture', 'blog.navigationBuilder.showPublishedInFuture', [
			0 => $this->t('default.form.no'),
			1 => $this->t('default.form.yes'),
		])->setDefaultValue(0);
		$container->addSelect('showLocked', 'blog.navigationBuilder.showLocked', [
			0 => $this->t('default.form.no'),
			1 => $this->t('default.form.yes'),
		])->setDefaultValue(0);
		$container->addSelect('sort', 'blog.navigationBuilder.sort', $sort);

		return $container;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
	}

	/*******************************************************************************************************************
	 * ===========================  Route
	 */

	public function presenterActionCheck($presenter, $action)
	{
		if ($this->presenter == $presenter && $this->action == $action)
			return 1;
		if ($this->presenterArticleDetail == $presenter && $this->actionArticleDetail == $action)
			return 1;

		return false;
	}

	public function routerIn($navigation, $urlParams = [])
	{
		$urlParamsId = isset($urlParams['path']) ? explode('/', $urlParams['path']) : 0;

		if (count($urlParamsId) > 1) {
			list($articleId, $articleAlias) = explode('-', $urlParamsId[1], 2);

			$article = $this->articlesService->get($articleId);

			if ($article) {
				return [
					'presenter' => $this->presenterArticleDetail,
					'action'    => $this->actionArticleDetail,
					'id'        => $articleId,
				];
			}
		}

		return [
			'presenter' => $this->presenter,
			'action'    => $this->action,
			'id'        => $navigation->componentParams['category'],
		];
	}

	public function routerOut($navigation, &$urlParams)
	{
		if (isset($urlParams['action']) && $urlParams['action'] == $this->actionArticleDetail && isset($urlParams['id'])) {
			$article = $this->articlesService->getForNavigation($urlParams['id']);

			return ($navigation->isHomepage ? '' : '/' . $navigation->alias) . '/' . ($article ? $article['id'] . '-' . $article['alias'] : $urlParams['id']);
		}

		$urlParts = [];
		if (!$navigation->isHomepage)
			$urlParts[] = $navigation->alias;

		return '/' . implode('/', $urlParts);
	}
}