<?php declare(strict_types = 1);

namespace Blog\Model\Rss;

use Nette;
use Nette\InvalidStateException;
use Nette\Utils\DateTime;
use Nette\Utils\Strings;

class Generator
{
	use Nette\SmartObject;

	/** @var string */
	protected $title = '';

	/** @var string */
	protected $description = '';

	/** @var string */
	protected $link = '';

	/** @var string */
	protected $dateTimeFormat = 'Y-m-d H:i:s';

	/** @var callable|null */
	protected $dataSource = null;

	/** @var int */
	protected $shortenDescription = 150;

	/** @var DateTime */
	protected $lastBuildDate;

	/**
	 * @return string
	 */
	function getTitle()
	{
		return $this->title;
	}

	/**
	 * @param string $title
	 */
	function setTitle(string $title)
	{
		$this->title = $title;
	}

	/**
	 * @return string
	 */
	function getDescription()
	{
		return $this->description;
	}

	/**
	 * @param string $description
	 */
	function setDescription(string $description)
	{
		$this->description = $description;
	}

	/**
	 * @return string
	 */
	function getLink()
	{
		return $this->link;
	}

	/**
	 * @param string $link
	 */
	function setLink(string $link)
	{
		$this->link = $link;
	}

	/**
	 * @param callable $dataSource
	 */
	function setDataSource(callable $dataSource)
	{
		$this->dataSource = $dataSource;
	}

	/**
	 * @return int
	 */
	function getShortenDescription()
	{
		return $this->shortenDescription;
	}

	/**
	 * @param int $value
	 */
	function setShortenDescription(int $value)
	{
		$this->shortenDescription = $value;
	}

	/**
	 * @return string
	 */
	function getDateTimeFormat()
	{
		return $this->dateTimeFormat;
	}

	/**
	 * @param string $format
	 */
	function setDateTimeFormat(string $format)
	{
		$this->dateTimeFormat = $format;
	}

	/**
	 * @return callable
	 */
	function getLastBuildDate()
	{
		return $this->lastBuildDate;
	}

	/**
	 * @param DateTime $lastBuildDate
	 */
	function setLastBuildDate(DateTime $lastBuildDate)
	{
		$this->lastBuildDate = $lastBuildDate;
	}

	/**
	 * @return \SimpleXMLElement
	 * @throws \InvalidArgumentException
	 */
	function generate(): \SimpleXMLElement
	{
		$doc = new \DOMDocument();
		if (is_null($this->dataSource)) {
			throw new InvalidStateException('Data source for RSS generator is not set.');
		}
		$items   = call_user_func($this->dataSource);
		$channel = simplexml_load_file(__DIR__ . '/template.xml');

		$NS = [
			'szn' => 'https://www.seznam.cz',
		];
		$channel->registerXPathNamespace('szn', $NS['szn']);

		if ($this->link) {
			$channel->channel->link[0][0] = $this->link;
		}
		if ($this->getLastBuildDate())
			$channel->channel->lastBuildDate = $this->getLastBuildDate()->format($this->dateTimeFormat);
		if ($this->title) {
			$channel->channel->title[0][0] = $this->title;
		}
		if ($this->description) {
			$channel->channel->description[0][0] = $this->description;
		}

		/** @var RssChannelItem $item */
		foreach ($items as $item) {
			/** @var \SimpleXMLElement $i */
			$i = $channel->channel->addChild('item');
			$i->addChild('title', $item->getTitle());
			$i->addChild('link', $item->getLink());
			$i->addChild('guid', $item->getLink());
			$i->addChild('image', $item->getImage(), $NS['szn']);
			$i->addChild('pubDate', $item->getPubDate()->format($this->getDateTimeFormat()));

			$desc = $i->addChild('description');
			$node = dom_import_simplexml($desc);
			$no   = $node->ownerDocument;
			$node->appendChild($no->createCDATASection(trim(Strings::truncate($item->getDescription(), $this->getShortenDescription()))));
		}

		return $channel;
	}

	/**
	 * @return RssResponse
	 */
	function response()
	{
		try {
			return new RssResponse($this->generate());
		} catch (InvalidStateException | \InvalidArgumentException $e) {
			throw $e;
		}
	}
}