<?php declare(strict_types = 1);

namespace Blog\AdminModule\Components\Article;

use Blog\Model\Articles;
use Blog\Model\Categories;
use Blog\Model\ConfigOptions;
use Blog\Model\Entities\Article;
use Blog\Model\Entities\ArticleText;
use Blog\Model\Entities\Category;
use Blog\Model\OpenedArticlesService;
use Core\Components\FilesManager\IFilesManagerFactory;
use Core\FrontModule\Model\SeoContainer;
use Core\Model\Lang\Langs;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Gallery\AdminModule\Components\Image\IImagesZoneFactory;
use Gallery\AdminModule\Components\Image\ImagesZone;
use Gallery\Model\Entities\Album;
use Nette\Forms\Form;
use Nette\Http\IResponse;
use Nette\Http\Request;
use Nette\Http\Session;
use Nette\Http\SessionSection;
use Nette\Utils\ArrayHash;
use Nette\Utils\DateTime;
use Tags\Model\Tags;
use Users\Model\Entities\Acl;
use Users\Model\Entities\Role;
use Users\Model\Entities\User;
use Users\Model\Users;

class ArticleForm extends BaseControl
{
	/** @var Article */
	public $article;

	/** @var ConfigOptions */
	protected $configOptions;

	/** @var Categories */
	private $categoriesService;

	/** @var Articles */
	private $articlesService;

	/** @var Tags */
	private $tagsService;

	/** @var IImagesZoneFactory */
	private $imagesZoneFactory;

	/** @var IFilesManagerFactory */
	private $filesManagerFactory;

	/** @var OpenedArticlesService */
	protected $openedArticlesService;

	/** @var Request */
	protected $httpRequest;

	/** @var Session */
	protected $session;

	/** @var SessionSection */
	protected $sessionSection;

	/** @var SeoContainer */
	protected $seoContainerService;

	public function __construct(ConfigOptions $configOptions, Articles $articles, Categories $categories, Tags $tags, IFilesManagerFactory $filesManagerFactory, IImagesZoneFactory $imagesZoneFactory,
	                            OpenedArticlesService $openedArticlesService, Request $httpRequest, Session $session, SeoContainer $seoContainer)
	{
		$this->configOptions         = $configOptions;
		$this->articlesService       = $articles;
		$this->categoriesService     = $categories;
		$this->tagsService           = $tags;
		$this->filesManagerFactory   = $filesManagerFactory;
		$this->imagesZoneFactory     = $imagesZoneFactory;
		$this->openedArticlesService = $openedArticlesService;
		$this->httpRequest           = $httpRequest;
		$this->session               = $session;
		$this->sessionSection        = $session->getSection('unsavedArticles');
		$this->seoContainerService   = $seoContainer;
	}

	public function render()
	{
		$presenter = $this->getPresenter();

		$opened = $this->article ? $this->openedArticlesService->checkArticle($this->article->getId()) : false;
		if ($opened && $opened->user->getId() != $presenter->getUser()->getId()) {
			echo $this->t('blog.article.isEditingBy', null, ['user' => $opened->user->name]) . ' - ' . $opened->lastActivity->format('H:i:s');

			return;
		}

		if ($this->article)
			$this->openedArticlesService->openArticle($this->article, $presenter->getUser()->getIdentity());

		$this->template->unsavedArticleExists = $this->getSavedData() ? true : false;
		$this->template->article              = $this->article;
		$this->template->setFile(__DIR__ . '/ArticleForm.latte');
		$this->template->render();
	}

	protected function attached($presenter)
	{
		parent::attached($presenter);
		if ($this->article)
			$this->template->article = $this->article;
		if (!$this['form']['author']->value)
			$this['form']['author']->setDefaultValue($this->getPresenter()->getUser()->getId());
	}

	protected function createComponentForm()
	{
		$form = $this->createForm();

		$categories = [null => $this->t('blog.articleForm.chooseCategory')];
		foreach ($this->categoriesService->getAll() as $cat)
			$categories[$cat->getId()] = $cat->getText()->getTitle();

		$tags = [];
		foreach ($this->tagsService->getAvailableTags() as $tag)
			$tags[$tag->getId()] = $tag->title;

		$roles = [];
		foreach ($this->em->getRepository(Acl::class)->createQueryBuilder('acl')
			         ->select('IDENTITY(acl.role) as role')
			         ->join('acl.privilege', 'pr', 'WITH', 'pr.name IN (:privilegeName)')
			         ->join('acl.resource', 'res', 'WITH', 'res.name IN (:resourceName)')
			         ->andWhere('acl.allowed = 1')
			         ->setParameters([
				         'privilegeName' => ['access', 'all'],
				         'resourceName'  => ['Blog:Admin', 'all'],
			         ])->getQuery()->getScalarResult() as $row)
			$roles[] = $row['role'];

		$users = [];
		foreach ($this->em->getRepository(User::class)->createQueryBuilder('u')
			         ->select('u.id, u.name, u.lastname')
			->join('u.roles', 'r', 'WITH', 'r.id IN (:roles)')->setParameter('roles', $roles)
			         ->orderBy('u.name', 'ASC')->getQuery()->getScalarResult() as $row)
			$users[$row['id']] = trim($row['lastname'] . ' ' . $row['name']);

		$form->addGroup('default.form.general');
		$form->addText('title', 'blog.articleForm.title')->setMaxLength(255)->setIsMultilanguage();
		$form->addText('alias', 'blog.articleForm.alias')->setMaxLength(255)->setIsMultilanguage();
		$form->addSelect('category', $this->t('blog.articleForm.category'), $categories)->setTranslator(null);
		$form->addSelect('author', $this->t('blog.articleForm.author'), $users)->setTranslator(null)->setIsMultilanguage();
		$form->addBool('isPublished', 'blog.articleForm.isPublished')->setDefaultValue(0)->setIsMultilanguage();
		$form->addTags('tags', 'blog.articleForm.tags', function($q) {
			$tags = [];
			foreach ($this->tagsService->getAvailableTags() as $tag)
				$tags[] = ['key' => $tag->getId(), 'value' => $tag->title];

			return $tags;
		}, function($value) {
			return $this->tagsService->createTag($value);
		});
		$form->addTextArea('introtext', 'blog.articleForm.introtext')->setMaxLength(255)->setIsMultilanguage();
		$form->addEditor('fulltext', 'blog.articleForm.fulltext')->setToolbar('Blog')
			->setAlbumImagesCallback([$this, 'albumImagesCallback'])->setDisableAutoP(false)->setIsMultilanguage();
		$form->addDateTimePicker('publishUp', 'blog.articleForm.publishUp')->setDefaultValue(new DateTime())->setRequired();
		$form->addDateTimePicker('publishDown', 'blog.articleForm.publishDown');
		$form->addText('source', 'blog.articleForm.source');
		$form->addTextArea('footer', 'blog.articleForm.footer');

		$form->addHidden('articleId');
		$form->addHidden('preparedAlbumId');

		$form->addComponent($this->seoContainerService->getContainer(true), 'seo');

		$form->addSubmit('submitAndClose', 'default.saveAndClose');
		$form->addSubmit('submit', 'default.save');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		$presenter = $this->getPresenter();
		$user      = $presenter->getUser();
		try {
			$langValues = $form->convertMultilangValuesToArray();
			/** @var ArticleText[] $texts */
			$texts = [];

			if ($this->article) {
				$article = $this->article;
				$texts   = $article->getTexts()->toArray();
			} else {
				$article = new Article();
			}

			foreach ($langValues as $l => $v) {
				if (!isset($texts[$l]) && $v['title'])
					$texts[$l] = new ArticleText($article, $v['title'], $l);

				if ($v['title'] == '' || $texts[$l]->getTitle() == '') {
					if ($texts[$l])
						$this->em->remove([$texts[$l]]);
					unset($texts[$l]);
					continue;
				}

				$texts[$l]->setTitle($v['title'])
					->setAlias($v['alias'] ?: $v['title']);
				$texts[$l]->isPublished = $v['isPublished'];
				$texts[$l]->introtext   = $v['introtext'];
				$texts[$l]->fulltext    = $v['fulltext'];
				$texts[$l]->createdBy   = $v['created'];
				$texts[$l]->modifiedBy  = $user->getIdentity();

				$this->em->persist($texts[$l]);
			}

			$article->setTexts($texts);
			$article->category    = $values->category ? $this->em->getReference(Category::class, $values->category) : null;
			$article->tags        = $values->tags ? array_map(function($id) {
				return $this->tagsService->getReference((int) $id);
			}, explode(',', $values->tags)) : [];
			$article->publishUp   = $values->publishUp;
			$article->publishDown = $values->publishDown;
			$article->params      = [
				'source' => $values->source,
				'footer' => $values->footer,
			];

			$this->em->persist($article);
			$this->em->flush();

			$this->article = $article;
			unset($this->sessionSection->{'article_' . $article->getId()});
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}

		return true;
	}

	public function setArticle($id)
	{
		$this->article = $this->em->getRepository(Article::class)->find($id);

		if (!$this->article)
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);

		$form = $this['form'];
		$a    = $this->article;
		$d    = [];
		$form->setDefaults([
			'category'        => $a->category ? $a->category->getId() : null,
			'publishUp'       => $a->publishUp,
			'publishDown'     => $a->publishDown,
			'articleId'       => $a->getId(),
			'preparedAlbumId' => $a->getGallery() ? $a->getGallery()->getId() : '',
		]);
		foreach ($a->getTexts() as $l => $text) {
			$d['title'][$l]       = $text->getTitle();
			$d['alias'][$l]       = $text->getAlias();
			$d['isPublished'][$l] = $text->isPublished;
			$d['introtext'][$l]   = $text->introtext;
			$d['fulltext'][$l]    = $text->fulltext;

			if ($text->createdBy && array_key_exists($text->createdBy->getId(), $form['author']->getItems()))
				$d['author'][$l] = $text->createdBy->getId();
		}

		$form['tags']->setDefaultValue(array_map(function($arr) {
			return ['key' => $arr->getId(), 'value' => $arr->title];
		}, $a->tags->toArray()));

		//			if ($a->getGallery()) {
		//				$roots = [];
		//				foreach ($a->getGallery()->getImages() as $image)
		//					$roots[$image->path] = $image->path;
		//				foreach ($roots as $root)
		//					$form['fulltext']->addRoot($root);
		//			}

		if (is_array($a->params))
			foreach ($a->params as $k => $v) {
				if (isset($form[$k])) {
					$type = $form[$k]->getOption('type');

					if ($type == 'select')
						if (!array_key_exists($v, $form[$k]->getItems()))
							continue;
					$form[$k]->setDefaultValue($v);
				}
			}

		$this->seoContainerService->setDefaultsFromEntity($form['seo'], $a->getTexts()->toArray());
		$form->setDefaults($d);
	}

	private function getSavedData()
	{
		return $this->article ? $this->sessionSection->{'article_' . $this->article->getId()} : false;
	}

	public function albumImagesCallback()
	{
		return 'preparedAlbumId';
	}

	/*******************************************************************************************************************
	 * =================  Handle
	 */

	// TODO odstranit
	public function handlePreSave()
	{
		$presenter = $this->getPresenter();
		$data      = $this->httpRequest->getPost('formData');
		if ($this->article || !$data)
			$presenter->sendPayload();

		bdump($data);
		die();

		try {
			$article            = new Article($data['title'], $presenter->getUser()->getIdentity());
			$article->publishUp = new DateTime();
			$this->em->persist($article);
			$this->em->flush();
			$presenter->payload->editUrl = $presenter->link('edit', [$article->getId()]);
			$presenter->payload->article = [
				'id'    => $article->getId(),
				'alias' => $article->alias,
			];
			$presenter->flashMessageSuccess('blog.article.articleCreated');
		} catch (\Exception $e) {
			$presenter->payload->error = true;
			$presenter->flashMessageDanger('blog.article.articleCreatingError');
		}

		$presenter->redrawControl('flashes');
	}

	public function handleSaveProgress()
	{
		$dataRaw = $this->httpRequest->getPost('formData');
		$data    = [];

		if (!$dataRaw)
			$this->getPresenter()->sendPayload();

		foreach ($dataRaw as $v) {
			if (in_array($v, ['_token_', '_do']))
				continue;
			$data[$v['name']] = $v['value'];
		}

		if (!$this->article)
			$this->article = $this->articlesService->getReference($data['articleId']);

		if (!$this->article)
			$this->getPresenter()->sendPayload();

		$this->sessionSection->{'article_' . $this->article->getId()} = $data;
		$this->getPresenter()->flashMessageSuccess('blog.article.autoSaveSuccess');
		$this->getPresenter()->redrawControl('flashes');
	}

	public function handleLoadUnsaved()
	{
		if (!$this->article || !($data = $this->sessionSection->{'article_' . $this->article->getId()}))
			return;

		$presenter                = $this->getPresenter();
		$presenter->payload->data = $data;
		$presenter->flashMessageSuccess('blog.article.unsavedDataLoaded');
		$presenter->redrawControl('flashes');
	}

	public function handleClearUnsaved()
	{
		if ($this->article) {
			unset($this->sessionSection->{'article_' . $this->article->getId()});
		}

		$this->getPresenter()->flashMessageSuccess('blog.article.unsavedDataCleared');
		$this->getPresenter()->redrawControl('flashes');
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentFilesManager()
	{
		return $this->filesManagerFactory->create();
	}

	protected function createComponentImagesZone()
	{
		$control = $this->imagesZoneFactory->create();

		if ($this->article && $this->article->gallery) {
			$control->setAlbum($this->article->gallery->getId());
		}

		$control->onEmpty[] = function($control) {
			/** @var ImagesZone $control */
			$dataRaw = $this->httpRequest->getPost('formData');
			$data    = [];

			if (!$dataRaw)
				return false;

			foreach ($dataRaw as $row) {
				$data[$row['name']] = $row['value'];
			}

			$this->article = $this->articlesService->getReference($data['articleId']);

			if (!$this->article) {
				$article            = new Article($data['title'], $this->getPresenter()->getUser()->getIdentity());
				$article->publishUp = DateTime::createFromFormat('j. n. Y H:i:s', $data['publishUp']);
				$this->em->persist($article);
				$this->em->flush();
				$this->article = $article;
			}

			if (!$this->article->gallery) {
				$album        = new Album(UPLOADS_PATH . '/gallery');
				$album->title = $data['title'];
				$this->em->persist($album);
				$this->em->flush();

				$control->setAlbum($album->getId());
				$this->article->gallery = $album;

				$this->em->persist($this->article);
				$this->em->flush();
			}

			$control->extendedResponseOnEmpty['articleId']    = $this->article->getId();
			$control->extendedResponseOnEmpty['articleTitle'] = $this->article->getText()->getTitle();
			$control->extendedResponseOnEmpty['articleAlias'] = $this->article->getText()->getAlias();
		};

		return $control;
	}
}
