<?php declare(strict_types = 1);

namespace Blog\FrontModule\Components;

use Blog\FrontModule\Model\Articles;
use Blog\Model\ArticlesFacade;
use Blog\Model\Categories;
use Blog\Model\ConfigOptions;
use Blog\Model\Entities\Article;
use Core\FrontModule\Components\StructureData\IDataSchemaFactory;
use Core\Model\UI\BaseControl;
use Gallery\Model\Images;
use Navigations\Model\Navigations;
use Nette\Caching\Cache;
use Nette\Utils\Paginator;
use Spatie\SchemaOrg\Schema;
use function SimpleHtmlDom\str_get_html;

class ArticlesList extends BaseControl
{
	/** @var array */
	protected $categories;

	/** @var string */
	protected $sort;

	/** @var bool */
	protected $showLocked;

	/** @var Article[] */
	protected $articles;

	/** @var int */
	protected $articlesCount;

	/** @var Articles */
	protected $articlesService;

	/** @var Categories */
	protected $categoriesService;

	/** @var Images */
	protected $imagesService;

	/** @var ArticlesFacade */
	protected $articlesFacade;

	protected $activeNavigation;

	/** @var ConfigOptions */
	protected $blogConfig;

	/** @var int */
	public $page = 1;

	public $limit = 8;
	public $start = 0;
	public $paginatorEnabled = true;

	public function __construct(ConfigOptions $config, Articles $articles, Categories $categories, Images $images, ArticlesFacade $articlesFacade)
	{
		$this->articlesService   = $articles;
		$this->categoriesService = $categories;
		$this->imagesService     = $images;
		$this->articlesFacade    = $articlesFacade;
		$this->blogConfig        = $config;
		$this->limit             = $config->get('articlesListLimit');
	}

	/**
	 * Po připojení k rodiči se nastaví základní data
	 *
	 * @param \Nette\ComponentModel\IComponent $presenter
	 */
	protected function attached($presenter)
	{
		parent::attached($presenter);
		$this->activeNavigation = $this->getPresenter()->activeNavigation;
		$this->cache            = new Cache($this->cacheStorage, Articles::CACHE_NAMESPACE);

		$this->sort       = $this->activeNavigation->componentParams['sort'] ?? 'publishUp';
		$this->showLocked = (bool)($this->activeNavigation->componentParams['showLocked'] ?? false);

		$this->page = $this->getParameter('page', 1);
	}

	public function render()
	{
		$this->template->url              = trim($this->getPresenter()->baseUrl, '/');
		$this->template->articlesCount    = $this->getArticlesCount();
		$this->template->limit            = $this->limit;
		$this->template->start            = $this->start;
		$this->template->paginatorEnabled = $this->paginatorEnabled;
		$this->template->articles         = $this->getArticles();
		$this->template->category         = $this->getCategory();

		$paginator = null;
		if ($this->paginatorEnabled) {
			$paginator = new Paginator();
			$paginator->setItemCount($this->getArticlesCount());
			$paginator->setItemsPerPage($this->limit);
			$paginator->setPage($this->page);
		}
		$this->template->paginator = $paginator;

		$this->template->render($this->getTemplateFile());
	}

	public function handlePaginator($page)
	{
		$this->page  = $page;
		$this->start = $page > 1 ? $this->limit * ($page - 1) : 0;
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	/*******************************************************************************************************************
	 * ============================== Get / Set
	 */

	/**
	 * Načtení článků které se zobrazí na stránce
	 *
	 * @return Article[]|null
	 */
	protected function getArticles()
	{
		if (!$this->articles) {
			$this->articles = $this->articlesService->getAll($this->categories, $this->start, $this->limit, $this->sort, $this->showLocked);
		}

		return $this->articles;
	}

	/**
	 * Vrací categorii popřípadě kategorie listovaných článků
	 * @return array|mixed
	 */
	protected function getCategory()
	{
		if (is_array($this->categories) && count($this->categories) == 1)
			return $this->categoriesService->get($this->categories[0]);

		//TODO tohle vraci jen IDCKA dopsat metodu do modelove tridy Categories
		return $this->categories;
	}

	/**
	 * Ziskání počtu zveřejněných článků
	 *
	 * @return int
	 */
	public function getArticlesCount()
	{
		if (!$this->articlesCount) {
			$this->articlesCount = $this->articlesService->getAllCount($this->categories, $this->showLocked);
		}

		return $this->articlesCount;
	}

	/**
	 * @param $categories
	 */
	public function setCategories($categories)
	{
		$this->categories = $categories;
	}

	/**
	 * @param int $limit
	 */
	public function setLimit(int $limit)
	{
		$this->limit = $limit;
	}
}
