<?php declare(strict_types = 1);

namespace Blog\Model\Entities;

use Core\Model\Entities\TTranslateListener;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Gallery\Model\Entities\Album;
use Gallery\Model\Entities\Image;
use Gedmo\Mapping\Annotation as Gedmo;
use Kdyby\Doctrine\Entities\Attributes\Identifier;
use Kdyby\Doctrine\Entities\MagicAccessors;
use Nette\Utils\DateTime;
use Tags\Model\Entities\Tag;

/**
 * Class Article
 * @package Blog\Model\Entities
 *
 * @ORM\Table(name="blog__article")
 * @ORM\Entity
 * @ORM\EntityListeners({"ArticleListener", "Core\Model\Entities\TranslateListener"})
 */
class Article
{
	use Identifier;
	use MagicAccessors;
	use TTranslateListener;

	/**
	 * @var Category
	 * @ORM\ManyToOne(targetEntity="Category")
	 * @ORM\JoinColumn(name="category_id", referencedColumnName="id", nullable=true, onDelete="SET NULL")
	 */
	protected $category;

	/**
	 * @var Tag[]
	 * @ORM\ManyToMany(targetEntity="Tags\Model\Entities\Tag", fetch="EXTRA_LAZY")
	 * @ORM\JoinTable(name="blog__article_tags",
	 *     joinColumns={@ORM\JoinColumn(name="article_id", referencedColumnName="id", onDelete="CASCADE")},
	 *     inverseJoinColumns={@ORM\JoinColumn(name="tag_id", referencedColumnName="id", onDelete="CASCADE")}
	 *     )
	 */
	protected $tags;

	/**
	 * @var string
	 * @ORM\Column(name="image", type="string", length=255, nullable=true)
	 */
	protected $image;

	/**
	 * @var string
	 * @ORM\Column(name="video", type="string", length=255, nullable=true)
	 */
	protected $video;

	/**
	 * @var Album
	 * @ORM\OneToOne(targetEntity="Gallery\Model\Entities\Album")
	 * @ORM\JoinColumn(name="gallery_id", referencedColumnName="id", onDelete="SET NULL")
	 */
	protected $gallery;

	/**
	 * @var DateTime
	 * @ORM\Column(name="publish_up", type="datetime", nullable=true)
	 */
	protected $publishUp;

	/**
	 * @var DateTime
	 * @ORM\Column(name="publish_down", type="datetime", nullable=true)
	 */
	protected $publishDown;

	/**
	 * @var string
	 * @ORM\Column(name="featured", type="string", length=60, nullable=true)
	 */
	private $featured;

	/**
	 * @var array
	 * @ORM\Column(name="params", type="array", nullable=true)
	 */
	public $params;

	/**
	 * @var int
	 * @ORM\Column(name="shows", type="integer", options={"unsigned"=true, "default"=0})
	 */
	public $shows;

	/**
	 * @var ArticleText[]
	 * @ORM\OneToMany(targetEntity="ArticleText", mappedBy="article", indexBy="lang")
	 */
	protected $texts;

	public function __construct()
	{
		$this->texts = new ArrayCollection();
		$this->shows = 0;
	}

	public function setTexts(array $texts): self
	{
		$this->texts = new ArrayCollection($texts);

		return $this;
	}

	public function getText(?string $lang = null): ?ArticleText { return $this->texts[$lang ?: $this->locale] ?? null; }

	/** @return ArticleText[]|ArrayCollection */
	public function getTexts() { return $this->texts; }

	public function setId($id)
	{
		$this->id = $id;
	}

	public function getCategory()
	{
		return $this->category;
	}

	/******
	 * === Image
	 */

	/** @return Image */
	public function getImage() { return $this->gallery && $this->gallery->getCoverImage() ? $this->gallery->getCoverImage() : null; }

	/** @return string */
	public function getImageFile() { return $this->getImage() ? $this->getImage()->getFilePath() : null; }

	/** @return Album */
	public function getGallery() { return $this->gallery; }

	/*******
	 * === Featured
	 */

	public function getFeatured() { return $this->featured; }

	public function getIsFeatured() { return $this->featured ? true : false; }

	public function setFeatured($featured)
	{
		$this->featured = $featured;

		return $this;
	}

	/******
	 * === Tags
	 */

	public function setTags(array $tags)
	{
		$this->tags = new ArrayCollection($tags);
	}

	/**
	 * @return array
	 */
	public function getTags() { return $this->tags; }

	public function getTagsId()
	{
		return $this->tags->map(function($tag) { return $tag->getId(); })->toArray();
	}
}
