<?php declare(strict_types = 1);

namespace Blog\Model\Navigation;

use Blog\FrontModule\Model\Articles;
use Blog\FrontModule\Model\Categories;
use Blog\Model\ConfigOptions;
use Blog\Model\Entities\Article;
use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Dao\SiteMapUrl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Navigations\Model\Providers\INavigationItem;
use Nette\Caching\Cache;
use Nette\Utils\ArrayHash;
use Pages\FrontModule\Model\TemplatePages;

class Blog extends BaseNavigation implements INavigationItem
{
	protected $title                  = 'blog';
	protected $presenter              = 'Blog:Front:Default';
	protected $action                 = 'default';
	protected $presenterArticleDetail = 'Blog:Front:Articles';
	protected $actionArticleDetail    = 'detail';

	const CACHE_ARTICLE = 'blogArticle';

	/** @var ConfigOptions */
	protected $configOptions;

	/** @var Articles */
	protected $articlesService;

	/** @var Categories */
	protected $categoriesService;

	/** @var TemplatePages */
	protected $templatePagesService;

	public function __construct(ConfigOptions $configOptions, Categories $categories, Articles $articles, TemplatePages $templatePages)
	{
		$this->configOptions        = $configOptions;
		$this->categoriesService    = $categories;
		$this->articlesService      = $articles;
		$this->templatePagesService = $templatePages;
	}

	/**
	 * @return BaseContainer
	 */
	public function getFormContainer()
	{
		$container = new BaseContainer();

		$categories = [];
		foreach ($this->categoriesService->getAll() as $cat) {
			/** @var \Blog\FrontModule\Model\Dao\Category $cat */
			$categories[$cat->getId()] = $cat->title;
		}

		$container->addMultiSelect('categories', 'blog.navigationBuilder.selectCategory', $categories)->getControlPrototype()->attrs['style'] = 'height: 100px';

		$templates = [null => ''];
		foreach ($this->templatePagesService->getEr()->findBy([]) as $t)
			$templates[$t->getId()] = $t->title;
		$container->addSelect('tpCategory', $this->t('blog.navigationBuilder.templatePageCategory'), $templates)
			->setTranslator(null);
		$container->addSelect('tpDetail', $this->t('blog.navigationBuilder.templatePageDetail'), $templates)
			->setTranslator(null);

		return $container;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
	}

	public function updateCacheDep(array &$cache, $urlParams)
	{
		if ($urlParams['id']) {
			$cache[Cache::TAGS][] = 'article/' . $urlParams['id'];
			$cache[Cache::TAGS][] = 'articles/';
		}

		parent::updateCacheDep($cache, $urlParams);
	}

	/*******************************************************************************************************************
	 * ===========================  Route
	 */

	public function presenterActionCheck($presenter, $action)
	{
		if ($this->presenter == $presenter && $this->action == $action)
			return 1;
		if ($this->presenterArticleDetail == $presenter && $this->actionArticleDetail == $action)
			return 2;

		return false;
	}

	public function routerIn($navigation, $urlParams = [])
	{
		$params = explode('/', $urlParams['path']);

		if (isset($urlParams['path']) && count($params) > 1) {
			$id = explode('-', end($params), 2);

			$return = [
				'presenter' => $this->presenterArticleDetail,
				'action'    => $this->actionArticleDetail,
			];

			if (!is_numeric($id[0])) {
				$articleId = $this->articlesService->getPublishedIdByAlias(end($params), $urlParams['locale']);

				if ($articleId)
					$return['id'] = $articleId;
			} else {
				$return['id'] = $id[0];
			}

			if (isset($return['id']))
				return $return;
		}

		return [
			'presenter' => $this->presenter,
			'action'    => $this->action,
			'id'        => null,
		];
	}

	public function routerOut(DaoNavigationItem $navigation, &$urlParams): ?string
	{
		if (isset($urlParams['id'])) {
			$article = $this->articlesService->getForNavigation((int) $urlParams['id'], $urlParams['locale'] ?: $navigation->getLang());

			if ($article) {
				$articleUrl = $article['alias'];

				if (!$this->configOptions->get('removeArticleIdFromUrl'))
					$articleUrl = $article['id'] . '-' . $articleUrl;

				return '/' . $navigation->alias . ($article['category'] == 1 ? '' : '/' . $article['categoryAlias']) . '/' . $articleUrl;
			}
		}

		$urlParts = [];
		if (!$navigation->isHomepage)
			$urlParts[] = $navigation->alias;

		return '/' . implode('/', $urlParts);
	}

	/**
	 * @param null $navigation
	 * @param bool $siteMapData
	 *
	 * @return array
	 */
	public function getComponentChild(string $lang, $navigation = null, $siteMapData = false)
	{
		$arr = [];

		/** @var Article $article */
		foreach ($this->articlesService->getAll($navigation->componentParams['categories'] ?? null) as $article) {
			$data = [
				'action'    => $this->actionArticleDetail,
				'id'        => $article->getId(),
				'presenter' => $this->presenterArticleDetail,
			];

			if ($siteMapData) {
				$siteMapUrl = new SiteMapUrl();
				if ($article->getModified())
					$siteMapUrl->setModified($article->getModified());
				$siteMapUrl->setSeo($article->getSeo());

				if ($article->getGallery()) {
					foreach ($article->getGallery()->getImages() as $img)
						$siteMapUrl->addImage($img);
				}

				$data['siteMapData']['default'] = $siteMapUrl;
			}

			if ($navigation)
				$data['activeNavigation'] = $navigation;

			$arr[] = $data;
		}

		return $arr;
	}
}
