<?php declare(strict_types = 1);

namespace Blog\FrontModule\Model;

use Blog\Model\Entities\Category;
use Core\Model\Helpers\BaseFrontEntityService;
use Kdyby\Doctrine\QueryBuilder;
use Kdyby\Translation\Translator;

/**
 * Class Categories
 * @package Blog\FrontModule\Model
 */
class Categories extends BaseFrontEntityService
{
	/** @var \stdClass */
	protected $entityClass = Category::class;

	/** @var Translator */
	public $translator;

	public function __construct(Translator $translator)
	{
		$this->translator = $translator;
	}

	/**
	 * Kritéria pro publikované kategorie
	 *
	 * @param QueryBuilder $qb
	 * @param bool         $withLockedCategory
	 *
	 * @return QueryBuilder $qb
	 */
	public function setPublishedCriteria(QueryBuilder $qb, $withLockedCategory = false)
	{
		$qb->andWhere('c.isPublished = 1')->andWhere('c.lang IS NULL OR c.lang = :locale OR c.lang = \'\'')
			->setParameter('locale', $this->translator->getLocale());

		if (!$withLockedCategory) {
			$qb->andWhere('c.password IS NULL');
		}

		return $qb;
	}

	/**
	 * Získání zveřejněné kateogrie
	 *
	 * @param int  $id
	 * @param bool $withLockedCategory
	 *
	 * @return Category|null
	 */
	public function get($id, $withLockedCategory = false)
	{
		$qb = $this->getEr()->createQueryBuilder('c');
		$qb = $this->setPublishedCriteria($qb, $withLockedCategory);

		$qb->andWhere('c.id = :id')->setParameter('id', $id);

		return $qb->getQuery()->getOneOrNullResult();
	}

	/**
	 * Získání všech zveřejněných článků
	 *
	 * @param string|array $sort
	 * @param bool         $withLockedCategories
	 *
	 * @return Category[]|null
	 */
	public function getAll($sort = 'publishUp', $withLockedCategories = false)
	{
		$qb = $this->getEr()->createQueryBuilder('c');
		$qb = $this->setPublishedCriteria($qb, $withLockedCategories);

		// Pořadí
		if (is_string($sort)) {
			switch ($sort) {
				case 'publishUp':
					$qb->orderBy('c.publishUp', 'DESC');
					break;
				case 'title':
					$qb->orderBy('c.title', 'ASC');
					break;
			};
		} else if (is_array($sort)) {
			foreach ($sort as $k => $v) {
				$qb->addOrderBy($k, $v);
			}
		}

		return $qb->getQuery()->getResult();
	}
}