<?php declare(strict_types = 1);

namespace Blog\FrontModule\Presenters;

use Blog\FrontModule\Components\Article\IArticleDetailFactory;
use Blog\FrontModule\Components\IArticlesListFactory;
use Blog\Model\ArticleHits;
use Blog\FrontModule\Model\Articles;
use Blog\Model\Entities\Article;
use Facebook\FrontModule\Components\ILikeControlFactory;
use Nette\Utils\Strings;
use Spatie\SchemaOrg\Schema;

final class ArticlesPresenter extends BasePresenter
{
	/** @var Articles */
	protected $articlesService;

	/** @var Article */
	protected $article;

	/** @var ArticleHits */
	protected $articleHitsService;

	public function __construct(Articles $articles, ArticleHits $articleHits)
	{
		$this->articlesService    = $articles;
		$this->articleHitsService = $articleHits;
	}

	/*******************************************************************************************************************
	 * ==================================   Renders
	 */

	public function renderDetail()
	{
		$this['meta']
			->setMeta('title', $this->article->title)
			->setMeta('og:title', $this->article->title, 'property')
			->setMeta('og:type', 'article', 'property')
			->setMeta('og:description', $this->article->getIntrotext(), 'property')
			->setMeta('og:url', $this->link('//detail', $this->article->getId()), 'property');

		if ($this->article->getCreatedBy())
			$this['meta']
				->setMeta('author', $this->article->getCreatedBy()->name)
				->setMeta('og:author', $this->article->getCreatedBy()->name, 'property');

		if ($this->article->getGallery()) {
			$gallery = $this->article->getGallery();

			$this['meta']
				->setMeta('og:image', rtrim($this->baseUrl, '/') . $gallery->getCoverImage()->getFilePath(), 'property')
				->setMeta('image', rtrim($this->baseUrl, '/') . $gallery->getCoverImage()->getFilePath());
		}
	}

	/*******************************************************************************************************************
	 * ==================================   Actions
	 */
	public function actionDetail($id)
	{
		/** @var Article $article */
		$this->article = $this->articlesService->get($id);

		if (!$this->article)
			$this->error($this->t('blog.article.notFound'));

		$this->article->preparedFulltext = $this->articlesService->processFulltext($this->article);

		$this['articleDetail']->setArticle($this->article);

		$article = $this->article;
		$schema  = Schema::article()
			->headline(Strings::truncate($article->title, 110))
			->author(Schema::person()->name($article->createdBy->name))
			->genre($article->category->title)
			->wordCount(strlen(strip_tags($article->getFulltext())))
			->url($this->link('//this'))
			->dateCreated($article->publishUp->format('Y-m-d'))
			->datePublished($article->publishUp->format('Y-m-d'))
			->description($article->introtext)
			->articleBody(strip_tags($article->getFulltext()));

		if ($article->modified)
			$schema->dateModified($article->modified->format('Y-m-d'));

		if ($article->getGallery()) {
			$schema->image($article->getGallery()->getCoverImage() ? $this->baseUrl . ltrim($article->getGallery()->getCoverImage()->getFilePath(), '/') : '');
		}

		$this['dataSchemaControl']->addSchema($schema);

		$seo = $article->getSeo();
		if (!isset($seo['title']))
			$seo['title'] = $article->title . ' $separator $siteName';
		$this->seoDataFromEntity     = $seo;
		$this->template->articleLink = $this->link('//:Blog:Front:Articles:detail', [$article->getId()]);
		$this->template->article     = $article;
	}

	public function actionPreview($articleId, $hash)
	{
		if (!$articleId || !$hash || md5($articleId . md5($articleId . 'article')) != $hash)
			$this->error();

		$article = $this->article = $this->articlesService->getArticle($articleId);
		$this['articleDetail']->setArticle($this->article->getId());
		$this->articlesService->getPreparedText($article, [$this, 'link']);
		$this['title']->setTitle($article->title);
		$this['meta']->setMeta('robots', 'noindex,nofollow');
		$this->template->categoryAlias = Strings::webalize($article->category->title);
		$this->template->articleLink   = $this->link('//:Blog:Front:Articles:detail', [$article->getId()]);
		$this->template->related       = [];
	}

	public function actionAddShow($id)
	{
		if ($this->isAjax()) {
			$this->articleHitsService->addShow($id);
		}
		$this->terminate();
	}

	/*******************************************************************************************************************
	 * ==================================   Components
	 */

	//	protected function createComponentArticlesList(IArticlesListFactory $factory)
	//	{
	//		$control = $factory->create();
	//		$control->setCategories([$this->article->category->getId()]);
	//
	//		return $control;
	//	}

	protected function createComponentArticleDetail(IArticleDetailFactory $factory)
	{
		$control = $factory->create();

		return $control;
	}

	protected function createComponentLikeControl(ILikeControlFactory $factory)
	{
		return $factory->create();
	}
}
