<?php declare(strict_types = 1);

namespace Blog\AdminModule\Components\Article;

use Blog\Model\Articles;
use Blog\Model\Categories;
use Blog\Model\ConfigOptions;
use Blog\Model\Entities\Article;
use Blog\Model\Entities\Category;
use Blog\Model\OpenedArticlesService;
use Core\Components\FilesManager\IFilesManagerFactory;
use Core\FrontModule\Model\SeoContainer;
use Core\Model\Lang\Langs;
use Core\Model\UI\BaseControl;
use Gallery\AdminModule\Components\Image\IImagesZoneFactory;
use Gallery\AdminModule\Components\Image\ImagesZone;
use Gallery\Model\Entities\Album;
use Nette\Forms\Form;
use Nette\Http\IResponse;
use Nette\Http\Request;
use Nette\Http\Session;
use Nette\Http\SessionSection;
use Nette\Utils\ArrayHash;
use Nette\Utils\DateTime;
use Tags\Model\Tags;
use Users\Model\Entities\User;
use Users\Model\Users;

class ArticleForm extends BaseControl
{
	/** @var Article */
	public $article;
	
	/** @var ConfigOptions */
	protected $configOptions;

	/** @var Categories */
	private $categoriesService;

	/** @var Articles */
	private $articlesService;

	/** @var Tags */
	private $tagsService;

	/** @var IImagesZoneFactory */
	private $imagesZoneFactory;

	/** @var IFilesManagerFactory */
	private $filesManagerFactory;

	/** @var OpenedArticlesService */
	protected $openedArticlesService;

	/** @var Request */
	protected $httpRequest;

	/** @var Session */
	protected $session;

	/** @var SessionSection */
	protected $sessionSection;

	/** @var SeoContainer */
	protected $seoContainerService;

	public function __construct(ConfigOptions $configOptions, Articles $articles, Categories $categories, Tags $tags, IFilesManagerFactory $filesManagerFactory, IImagesZoneFactory $imagesZoneFactory,
	                            OpenedArticlesService $openedArticlesService, Request $httpRequest, Session $session, SeoContainer $seoContainer)
	{
		$this->configOptions         = $configOptions;
		$this->articlesService       = $articles;
		$this->categoriesService     = $categories;
		$this->tagsService           = $tags;
		$this->filesManagerFactory   = $filesManagerFactory;
		$this->imagesZoneFactory     = $imagesZoneFactory;
		$this->openedArticlesService = $openedArticlesService;
		$this->httpRequest           = $httpRequest;
		$this->session               = $session;
		$this->sessionSection        = $session->getSection('unsavedArticles');
		$this->seoContainerService   = $seoContainer;
	}

	public function render()
	{
		$presenter = $this->getPresenter();

		$opened = $this->article ? $this->openedArticlesService->checkArticle($this->article->getId()) : false;
		if ($opened && $opened->user->getId() != $presenter->getUser()->getId()) {
			echo $this->t('blog.article.isEditingBy', null, ['user' => $opened->user->name]) . ' - ' . $opened->lastActivity->format('H:i:s');

			return;
		}

		if ($this->article)
			$this->openedArticlesService->openArticle($this->article, $presenter->getUser()->getIdentity());

		$this->template->unsavedArticleExists = $this->getSavedData() ? true : false;
		$this->template->article              = $this->article;
		$this->template->setFile(__DIR__ . '/ArticleForm.latte');
		$this->template->render();
	}

	protected function attached($presenter)
	{
		parent::attached($presenter);
		if ($this->article)
			$this->template->article = $this->article;
		if (!$this['form']['author']->value)
			$this['form']['author']->setDefaultValue($this->getPresenter()->getUser()->getId());
	}

	protected function createComponentForm()
	{
		$form = $this->createForm();

		$categories = [null => $this->t('blog.articleForm.chooseCategory')];
		foreach ($this->categoriesService->getAll() as $cat)
			$categories[$cat->getId()] = $cat->title;

		$tags = [];
		foreach ($this->tagsService->getAvailableTags() as $tag)
			$tags[$tag->getId()] = $tag->title;

		$users = [];
		foreach ($this->em->getRepository(User::class)->findBy([], ['name' => 'ASC']) as $user)
			$users[$user->getId()] = $user->name;

		$form->addGroup('default.form.general');
		$form->addText('title', 'blog.articleForm.title')->setMaxLength(255)->setRequired();
		$form->addText('alias', 'blog.articleForm.alias')->setMaxLength(255);
		$form->addSelect('category', $this->t('blog.articleForm.category'), $categories)->setTranslator(null)->setRequired();
		$form->addSelect('author', $this->t('blog.articleForm.author'), $users)->setTranslator(null);
		$form->addBool('isPublished', 'blog.articleForm.isPublished')->setDefaultValue(0);
		$form->addLangsSelect('lang', 'default.lang');
		$form->addTags('tags', 'blog.articleForm.tags', function($q) {
			$tags = [];
			foreach ($this->tagsService->getAvailableTags() as $tag)
				$tags[] = ['key' => $tag->getId(), 'value' => $tag->title];

			return $tags;
		}, function($value) {
			return $this->tagsService->createTag($value);
		});
		$form->addTextArea('introtext', 'blog.articleForm.introtext');
		$form->addEditor('fulltext', 'blog.articleForm.fulltext')->setToolbar('Blog')
			->setAlbumImagesCallback([$this, 'albumImagesCallback'])->setDisableAutoP(false);
		//		$form->addFilesManager('image', 'blog.articleForm.image');
		//		$form->addFilesManager('video', 'blog.articleForm.video');
		$form->addDateTimePicker('publishUp', 'blog.articleForm.publishUp')->setDefaultValue(new DateTime())->setRequired();
		$form->addDateTimePicker('publishDown', 'blog.articleForm.publishDown');
		$form->addText('source', 'blog.articleForm.source');
		$form->addTextArea('footer', 'blog.articleForm.footer');

		$form->addHidden('articleId');
		$form->addHidden('preparedAlbumId');

		$form->addComponent($this->seoContainerService->getContainer(), 'seo');

		$form->addSubmit('submitAndClose', 'default.saveAndClose');
		$form->addSubmit('submit', 'default.save');

		if ($this->article) {
			$a = $this->article;
			$form->setDefaults([
				'title'           => $a->title,
				'alias'           => $a->alias,
				'category'        => $a->category ? $a->category->getId() : null,
				'isPublished'     => $a->isPublished,
				'introtext'       => $a->introtext,
				'fulltext'        => $a->fulltext,
				'publishUp'       => $a->publishUp,
				'publishDown'     => $a->publishDown,
				'articleId'       => $a->getId(),
				'preparedAlbumId' => $a->getGallery() ? $a->getGallery()->getId() : '',
			]);

			if ($a->getCreatedBy() && array_key_exists($a->createdBy->getId(), $form['author']->getItems()))
				$form['author']->setDefaultValue($a->createdBy->getId());

			if (array_key_exists($a->getLang(), $form['lang']->getItems()))
				$form['lang']->setDefaultValue($a->getLang());

			$form['tags']->setDefaultValue(array_map(function($arr) {
				return ['key' => $arr->getId(), 'value' => $arr->title];
			}, $a->tags->toArray()));

			//			if ($a->getGallery()) {
			//				$roots = [];
			//				foreach ($a->getGallery()->getImages() as $image)
			//					$roots[$image->path] = $image->path;
			//				foreach ($roots as $root)
			//					$form['fulltext']->addRoot($root);
			//			}

			if (is_array($a->params))
				foreach ($a->params as $k => $v) {
					if (isset($form[$k])) {
						$type = $form[$k]->getOption('type');

						if ($type == 'select')
							if (!array_key_exists($v, $form[$k]->getItems()))
								continue;
						$form[$k]->setDefaultValue($v);
					}
				}

			$this->seoContainerService->setDefaults($form['seo'], $a->getSeo());
		}

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(Form $form, ArrayHash $values)
	{
		try {
			$presenter = $this->getPresenter();
			$user      = $presenter->getUser();
			$createdBy = $this->em->getReference(User::class, $values->author);

			if ($this->article)
				$article = $this->article;
			else if ($values->articleId)
				$article = $this->articlesService->getArticle($values->articleId);
			else
				$article = new Article($values->title, $createdBy);

			$article->title = $values->title;
			$article->setAlias($values->alias);
			$article->setLang($values->lang);
			$article->setSeo($this->seoContainerService->getFormData($values->seo));
			$article->category    = $this->em->getRepository(Category::class)->find($values->category);
			$article->tags        = $values->tags ? array_map(function($id) {
				return $this->tagsService->get((int) $id);
			}, explode(',', $values->tags)) : [];
			$article->isPublished = $values->isPublished;
			$article->setIntrotext($values->introtext, $this->configOptions->get('introtextMaxLength'));
			$article->fulltext    = $values->fulltext;
			$article->modifiedBy  = $user->getIdentity();
			$article->publishUp   = $values->publishUp;
			$article->publishDown = $values->publishDown;
			$article->createdBy   = $createdBy;
			$article->params      = [
				'source' => $values->source,
				'footer' => $values->footer,
			];

			if ($values->introtext == '') {
				$article->setIntrotext($values->fulltext, $this->configOptions->get('introtextMaxLength'));
			}

			if (isset($values->deleted))
				$article->setDeleted((bool) $values->deleted);

			$this->em->persist($article);
			$this->em->flush();

			$this->article = $article;
			unset($this->sessionSection->{'article_' . $article->getId()});
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}
	}

	public function setArticle($id)
	{
		$this->article = $this->em->getRepository(Article::class)->find($id);

		if (!$this->article)
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
	}

	private function getSavedData()
	{
		return $this->article ? $this->sessionSection->{'article_' . $this->article->getId()} : false;
	}

	public function albumImagesCallback()
	{
		return 'preparedAlbumId';
	}

	/*******************************************************************************************************************
	 * =================  Handle
	 */
	public function handlePreSave()
	{
		$presenter = $this->getPresenter();
		$data      = $this->httpRequest->getPost('formData');
		if ($this->article || !$data)
			$presenter->sendPayload();

		try {
			$article            = new Article($data['title'], $presenter->getUser()->getIdentity());
			$article->publishUp = new DateTime();
			$this->em->persist($article);
			$this->em->flush();
			$presenter->payload->editUrl = $presenter->link('edit', [$article->getId()]);
			$presenter->payload->article = [
				'id'    => $article->getId(),
				'alias' => $article->alias,
			];
			$presenter->flashMessageSuccess('blog.article.articleCreated');
		} catch (\Exception $e) {
			$presenter->payload->error = true;
			$presenter->flashMessageDanger('blog.article.articleCreatingError');
		}

		$presenter->redrawControl('flashes');
	}

	public function handleSaveProgress()
	{
		$dataRaw = $this->httpRequest->getPost('formData');
		$data    = [];

		if (!$dataRaw)
			$this->getPresenter()->sendPayload();

		foreach ($dataRaw as $v) {
			if (in_array($v, ['_token_', '_do']))
				continue;
			$data[$v['name']] = $v['value'];
		}

		if (!$this->article)
			$this->article = $this->articlesService->getReference($data['articleId']);

		if (!$this->article)
			$this->getPresenter()->sendPayload();

		$this->sessionSection->{'article_' . $this->article->getId()} = $data;
		$this->getPresenter()->flashMessageSuccess('blog.article.autoSaveSuccess');
		$this->getPresenter()->redrawControl('flashes');
	}

	public function handleLoadUnsaved()
	{
		if (!$this->article || !($data = $this->sessionSection->{'article_' . $this->article->getId()}))
			return;

		$presenter                = $this->getPresenter();
		$presenter->payload->data = $data;
		$presenter->flashMessageSuccess('blog.article.unsavedDataLoaded');
		$presenter->redrawControl('flashes');
	}

	public function handleClearUnsaved()
	{
		if ($this->article) {
			unset($this->sessionSection->{'article_' . $this->article->getId()});
		}

		$this->getPresenter()->flashMessageSuccess('blog.article.unsavedDataCleared');
		$this->getPresenter()->redrawControl('flashes');
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentFilesManager()
	{
		return $this->filesManagerFactory->create();
	}

	protected function createComponentImagesZone()
	{
		$control = $this->imagesZoneFactory->create();

		if ($this->article && $this->article->gallery) {
			$control->setAlbum($this->article->gallery->getId());
		}

		$control->onEmpty[] = function($control) {
			/** @var ImagesZone $control */
			$dataRaw = $this->httpRequest->getPost('formData');
			$data    = [];

			if (!$dataRaw)
				return false;

			foreach ($dataRaw as $row) {
				$data[$row['name']] = $row['value'];
			}

			$this->article = $this->articlesService->getReference($data['articleId']);

			if (!$this->article) {
				$article            = new Article($data['title'], $this->getPresenter()->getUser()->getIdentity());
				$article->publishUp = DateTime::createFromFormat('j. n. Y H:i:s', $data['publishUp']);
				$this->em->persist($article);
				$this->em->flush();
				$this->article = $article;
			}

			if (!$this->article->gallery) {
				$album        = new Album(UPLOADS_PATH . '/gallery');
				$album->title = $data['title'];
				$this->em->persist($album);
				$this->em->flush();

				$control->setAlbum($album->getId());
				$this->article->gallery = $album;

				$this->em->persist($this->article);
				$this->em->flush();
			}

			$control->extendedResponseOnEmpty['articleId']    = $this->article->getId();
			$control->extendedResponseOnEmpty['articleTitle'] = $this->article->title;
			$control->extendedResponseOnEmpty['articleAlias'] = $this->article->alias;
		};

		return $control;
	}
}
