<?php declare(strict_types = 1);

namespace Blog\Model\Entities;

use Core\Model\Entities\TLang;
use Core\Model\Entities\TSeo;
use Core\Model\Lang\Langs;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Gallery\Model\Entities\Album;
use Gallery\Model\Entities\Image;
use Gedmo\Mapping\Annotation as Gedmo;
use Kdyby\Doctrine\Entities\Attributes\Identifier;
use Kdyby\Doctrine\Entities\MagicAccessors;
use Nette\Utils\DateTime;
use Nette\Utils\Strings;
use SimpleHtmlDom\simple_html_dom;
use Tags\Model\Entities\Tag;
use Users\Model\Entities\User;

/**
 * Class Article
 * @package Blog\Model\Entities
 *
 * @ORM\Table(name="blog__article", indexes={@ORM\Index(name="publish_idx", columns={"is_published"}),@ORM\Index(name="lang_idx", columns={"lang"})})
 * @ORM\Entity
 * @ORM\EntityListeners({"ArticleListener"})
 */
class Article
{
	use Identifier;
	use MagicAccessors;
	use TLang;
	use TSeo;

	/**
	 * @var string
	 * @ORM\Column(name="title", type="string", length=255)
	 */
	public $title;

	/**
	 * @var string
	 * @ORM\Column(name="alias", type="string", length=255)
	 */
	protected $alias;

	/**
	 * @var Category
	 * @ORM\ManyToOne(targetEntity="Category")
	 * @ORM\JoinColumn(name="category_id", referencedColumnName="id", nullable=true, onDelete="SET NULL")
	 */
	protected $category;

	/**
	 * @var Tag[]
	 * @ORM\ManyToMany(targetEntity="Tags\Model\Entities\Tag", fetch="EXTRA_LAZY")
	 * @ORM\JoinTable(name="blog__article_tags",
	 *     joinColumns={@ORM\JoinColumn(name="article_id", referencedColumnName="id", onDelete="CASCADE")},
	 *     inverseJoinColumns={@ORM\JoinColumn(name="tag_id", referencedColumnName="id", onDelete="CASCADE")}
	 *     )
	 */
	protected $tags;

	/**
	 * @var int
	 * @ORM\Column(name="is_published", type="smallint")
	 */
	protected $isPublished;

	/**
	 * @var string
	 * @ORM\Column(name="introtext", type="string", nullable=true)
	 */
	protected $introtext;

	/**
	 * @var string
	 * @ORM\Column(name="`text`", type="text", nullable=true)
	 */
	protected $fulltext;
	protected $preparedFulltext;

	/**
	 * @var string
	 * @ORM\Column(name="image", type="string", length=255, nullable=true)
	 */
	protected $image;

	/**
	 * @var string
	 * @ORM\Column(name="video", type="string", length=255, nullable=true)
	 */
	protected $video;

	/**
	 * @var Album
	 * @ORM\OneToOne(targetEntity="Gallery\Model\Entities\Album")
	 * @ORM\JoinColumn(name="gallery_id", referencedColumnName="id", onDelete="SET NULL")
	 */
	protected $gallery;

	/**
	 * @var DateTime
	 * @Gedmo\Timestampable(on="create")
	 * @ORM\Column(name="created", type="datetime")
	 */
	private $created;

	/**
	 * @var User
	 * @ORM\ManyToOne(targetEntity="\Users\Model\Entities\User")
	 * @ORM\JoinColumn(name="created_by", referencedColumnName="id", onDelete="SET NULL")
	 */
	protected $createdBy;

	/**
	 * @var DateTime
	 * @Gedmo\Timestampable(on="update")
	 * @ORM\Column(name="modified", type="datetime", nullable=true)
	 */
	private $modified;

	/**
	 * @var User
	 * @ORM\ManyToOne(targetEntity="\Users\Model\Entities\User")
	 * @ORM\JoinColumn(name="modified_by", referencedColumnName="id", nullable=true, onDelete="SET NULL")
	 */
	protected $modifiedBy;

	/**
	 * @var DateTime
	 * @ORM\Column(name="publish_up", type="datetime", nullable=true)
	 */
	protected $publishUp;

	/**
	 * @var DateTime
	 * @ORM\Column(name="publish_down", type="datetime", nullable=true)
	 */
	protected $publishDown;

	/**
	 * @var string
	 * @ORM\Column(name="featured", type="string", length=60, nullable=true)
	 */
	private $featured;

	/**
	 * @var array
	 * @ORM\Column(name="params", type="array", nullable=true)
	 */
	public $params;

	/**
	 * @var int
	 * @ORM\Column(name="shows", type="integer", options={"unsigned"=true, "default"=0})
	 */
	public $shows;

	public function __construct($title, $createdBy)
	{
		$this->title = $title;
		$this->setAlias($title);
		$this->createdBy   = $createdBy;
		$this->isPublished = 0;
		$this->tags        = new ArrayCollection();
		$this->shows       = 0;
		$this->seo         = [];
	}

	public function setId($id)
	{
		$this->id = $id;
	}

	public function setAlias($alias)
	{
		if ($alias)
			$this->alias = Strings::webalize($alias);
		else
			$this->alias = Strings::webalize($this->title);
	}

	public function getCreatedBy() { return $this->createdBy; }

	public function getCreated() { return $this->created; }

	public function getModified() { return $this->modified; }

	public function hasCompleteData()
	{
		return $this->title && $this->category;
	}

	public function canShow($params = [])
	{
		$now = new DateTime();

		$params                = (array) $params;
		$showPublishedInFuture = isset($params['showPublishedInFuture']) && $params['showPublishedInFuture'] == 1;

		return ($this->hasCompleteData() &&
			(($this->publishUp == null || $this->publishUp->getTimestamp() <= $now->getTimestamp()) || $showPublishedInFuture)
			&& ($this->publishDown == null || $this->publishDown->getTimestamp() >= $now->getTimestamp())
			&& $this->isPublished);
	}

	public function getCategory()
	{
		return $this->category;
	}

	/******
	 * === Image
	 */

	/** @return Image */
	public function getImage() { return $this->gallery && $this->gallery->getCoverImage() ? $this->gallery->getCoverImage() : null; }

	/** @return string */
	public function getImageFile() { return $this->getImage() ? $this->getImage()->getFilePath() : null; }

	/** @return Album */
	public function getGallery() { return $this->gallery; }

	/*******
	 * === Introtext
	 */

	public function setIntrotext($text, $maxLength = 160)
	{
		$this->introtext = Strings::truncate(trim(strip_tags(html_entity_decode($text))), $maxLength);
	}

	public function getIntrotext()
	{
		$str = str_replace('&nbsp;', ' ', $this->introtext);
		$str = trim($str);

		return $str;
	}

	/*******
	 * === Fulltext
	 */

	public function getFulltext() { return $this->fulltext; }

	public function setFulltext($text)
	{
		$this->fulltext = $text;

		if (!$this->introtext) {
			$this->setIntrotext($this->fulltext);
		}
	}

	public function getPreparedFulltext()
	{
		return $this->preparedFulltext ?: $this->fulltext;
	}

	/*******
	 * === Featured
	 */

	public function getFeatured() { return $this->featured; }

	public function getIsFeatured() { return $this->featured ? true : false; }

	public function setFeatured($featured)
	{
		$this->featured = $featured;

		return $this;
	}

	/******
	 * === Tags
	 */

	public function setTags(array $tags)
	{
		$this->tags = new ArrayCollection($tags);
	}

	/**
	 * @return array
	 */
	public function getTags() { return $this->tags; }

	public function getTagsId()
	{
		return $this->tags->map(function($tag) { return $tag->getId(); })->toArray();
	}
}
