<?php declare(strict_types = 1);

namespace Blog\Model\Navigation;

use Blog\FrontModule\Model\Articles;
use Blog\Model\Categories;
use Blog\Model\ConfigOptions;
use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Navigations\Model\Entities\Navigation;
use Navigations\Model\Providers\INavigationItem;
use Nette\Caching\Cache;
use Nette\Utils\ArrayHash;

class Category extends BaseNavigation implements INavigationItem
{
	protected $title                  = 'category';
	protected $presenter              = 'Blog:Front:Categories';
	protected $action                 = 'category';
	protected $presenterArticleDetail = 'Blog:Front:Articles';
	protected $actionArticleDetail    = 'detail';

	const CACHE_ARTICLE = 'blogArticle';

	/** @var ConfigOptions */
	protected $configOptions;

	/** @var Categories */
	protected $categoriesService;

	/** @var Articles */
	protected $articlesService;

	public function __construct(ConfigOptions $configOptions, Categories $categories, Articles $articles)
	{
		$this->configOptions     = $configOptions;
		$this->categoriesService = $categories;
		$this->articlesService   = $articles;
	}

	/**
	 * @return BaseContainer
	 */
	public function getFormContainer()
	{
		$container = new BaseContainer();

		$categories = [];
		foreach ($this->categoriesService->getAll() as $category)
			$categories[$category->getId()] = $category->title;

		$sort = [];
		foreach (['default', 'random'] as $v)
			$sort[$v] = $this->t('blog.navigationBuilder.sortTypes.' . $v);

		$container->addSelect('category', 'blog.navigationBuilder.category', $categories);
		$container->addSelect('showPublishedInFuture', 'blog.navigationBuilder.showPublishedInFuture', [
			0 => $this->t('default.form.no'),
			1 => $this->t('default.form.yes'),
		])->setDefaultValue(0);
		$container->addSelect('showLocked', 'blog.navigationBuilder.showLocked', [
			0 => $this->t('default.form.no'),
			1 => $this->t('default.form.yes'),
		])->setDefaultValue(0);
		$container->addSelect('sort', 'blog.navigationBuilder.sort', $sort);

		return $container;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
	}

	public function updateCacheDep(array &$cache, $urlParams)
	{
		$cache[Cache::TAGS][] = self::CACHE_ARTICLE . '/' . $urlParams['id'];

		parent::updateCacheDep($cache, $urlParams);
	}

	/*******************************************************************************************************************
	 * ===========================  Route
	 */

	public function presenterActionCheck($presenter, $action)
	{
		if ($this->presenter == $presenter && $this->action == $action)
			return 1;
		if ($this->presenterArticleDetail == $presenter && $this->actionArticleDetail == $action)
			return 1;

		return false;
	}

	public function routerIn($navigation, $urlParams = [])
	{
		$params = explode('/', $urlParams['path']);

		if (isset($urlParams['path']) && count($params) > 1) {
			$id = explode('-', end($params), 2);

			$return = [
				'presenter' => $this->presenterArticleDetail,
				'action'    => $this->actionArticleDetail,
			];

			if (!is_numeric($id[0])) {
				$articleId = $this->articlesService->getPublishedIdByAlias(end($params));

				if ($articleId)
					$return['id'] = $articleId;
			} else {
				$return['id'] = $id[0];
			}

			if (isset($return['id']))
				return $return;
		}

		return [
			'presenter' => $this->presenter,
			'action'    => $this->action,
			'id'        => $navigation->componentParams['category'],
		];
	}

	public function routerOut(DaoNavigationItem $navigation, &$urlParams): ?string
	{
		if (isset($urlParams['action']) && $urlParams['action'] == $this->actionArticleDetail && isset($urlParams['id'])) {
			$article = $this->articlesService->getForNavigation($urlParams['id']);

			if ($article) {
				$articleUrl = $article['alias'];

				if (!$this->configOptions->get('removeArticleIdFromUrl'))
					$articleUrl = $article['id'] . '-' . $articleUrl;

				return ($navigation->isHomepage ? '' : '/' . $navigation->alias) . '/' . $articleUrl;
			}
		}

		$urlParts = [];
		if (!$navigation->isHomepage)
			$urlParts[] = $navigation->alias;

		return '/' . implode('/', $urlParts);
	}
}
