<?php declare(strict_types = 1);

namespace Blog\AdminModule\Components\Category;

use Blog\Model\Articles;
use Blog\Model\Categories;
use Blog\Model\Entities\Category;
use Core\Components\Flashes\Flashes;
use Core\Model\UI\BaseControl;
use Nette\Utils\Html;

class CategoriesGrid extends BaseControl
{
	/** @var Categories */
	private $categoriesService;

	/** @var Articles */
	private $articlesService;

	public function __construct(Categories $categories, Articles $articles)
	{
		$this->categoriesService = $categories;
		$this->articlesService   = $articles;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentGrid()
	{
		$grid = $this->createGrid();
		$grid->setDefaultPerPage(50);

		$qb = $this->em->getRepository(Category::class)->createQueryBuilder('c')
			->addSelect('ct')
			->leftJoin('c.texts', 'ct');
		$grid->setDataSource($qb);

		// Columns
		$grid->addColumnLink('title', 'blog.categoriesGrid.title', 'Categories:edit', 'text.title');

		// Actions
		$grid->addAction('edit', '', 'Categories:edit')->setIcon('edit')->setBsType('primary');
		$grid->addAction('delete', '', 'delete!')->setIcon('times')->setBsType('danger')->addClass('ajax');

		return $grid;
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete($id)
	{
		$presenter = $this->getPresenter();
		if ($this->categoriesService->removeCategory($id))
			$presenter->flashMessage('blog.categoriesGrid.categoryRemoved', Flashes::FLASH_SUCCESS);
		else
			$presenter->flashMessage('blog.categoriesGrid.categoryRemoveFailed', Flashes::FLASH_DANGER);

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	/*******************************************************************************************************************
	 * =================  Grid renders
	 */

	/**
	 * @param Category $row
	 *
	 * @return Html
	 * @throws
	 */
	public function gridRenderTitle($row)
	{
		$articlesCount = $this->articlesService->getArticlesCountInCategoryByPublishing($row->getId());
		$html          = Html::el('');
		$html->addHtml(Html::el('a', ['href' => $this->getPresenter()->link('Categories:edit', $row->getId())])->setText($row->title));
		$html->addHtml(' ');
		$html->addHtml($span = Html::el('small', ['class' => 'text-secondary'])->setText('('));
		$span->addHtml(Html::el('span')->setText($this->t('default.published') . ' ' . $articlesCount[0]['count']));
		$span->addHtml(', ');
		$span->addHtml(Html::el('span')->setText($this->t('default.unPublished') . ' ' . $articlesCount[1]['count']));
		$span->addHtml(')');

		return $html;
	}
}
