<?php declare(strict_types = 1);

namespace Blog\AdminModule\Components\Article;

use Blog\Model\ArticleHits;
use Blog\AdminModule\Model\Articles;
use Blog\Model\Entities\Article;
use Blog\Model\Entities\Category;
use Blog\Model\OpenedArticlesService;
use Core\Components\Flashes\Flashes;
use Core\Model\UI\BaseControl;
use Doctrine\ORM\Query\Expr\Join;
use Gallery\Model\Entities\Album;
use Gallery\Model\Entities\Image;
use Kdyby\Doctrine\QueryBuilder;
use Nette\Utils\Html;
use Users\Model\Entities\User;

class ArticlesGrid extends BaseControl
{
	/** @var Articles */
	private $articlesService;

	/** @var ArticleHits */
	protected $articleHitsService;

	/** @var OpenedArticlesService */
	protected $openedArticlesService;

	/** @var array */
	protected $articleHits = [], $openedArticles = [];

	/** @var Image[]|array */
	protected $articleImages = [];

	public function __construct(Articles $articles, ArticleHits $articleHits, OpenedArticlesService $openedArticlesService)
	{
		$this->articlesService       = $articles;
		$this->articleHitsService    = $articleHits;
		$this->openedArticlesService = $openedArticlesService;
	}

	public function render()
	{
		$this->template->setFile($this->getTemplateFile());
		$this->template->render();
	}

	protected function createComponentGrid()
	{
		$grid = $this->createGrid();
		$grid->setTableClass('table-blog-articles');

//		$categories = $this->em->getRepository(Category::class)->findAll();
//		$users      = $this->em->getRepository(User::class)->findBy([], ['name' => 'ASC']);
		$qb = $this->em->getRepository(Article::class)->createQueryBuilder('a')
			->addSelect('at, c, ct')
			->leftJoin('a.texts', 'at')
			->leftJoin('a.category', 'c')
			->leftJoin('c.texts', 'ct');

		$categories = [];
		foreach ($this->em->getRepository(Category::class)->createQueryBuilder('c')
			         ->select('c.id, ct.title')
			         ->innerJoin('c.texts', 'ct', 'WITH', 'ct.lang = :lang')
			         ->setParameter('lang', $this->translator->getLocale())
			         ->getQuery()->getArrayResult() as $v)
			$categories[$v['id']] = $v['title'];

		$grid->setDataSource($qb);
		$grid->setDefaultSort(['id' => 'DESC']);
//		$grid->setOuterFilterRendering();

		$grid->getDataSource()->onDataLoaded[] = function($items) {
			/** @var Article[] $items */
			$ids      = [];
			$albumIds = [];
			$hits     = [];
			$images   = [];
			$opened   = [];
			foreach ($items as $item) {
				$ids[] = $item->getId();
				if ($item->getGallery())
					$albumIds[] = $item->getGallery()->getId();
			}

			$imagesRaw = $this->em->getRepository(Image::class)->createQueryBuilder('i')//->select('i.id, i.filename, a.id as albumId, a.basePath')
			->join(Album::class, 'a', 'WITH', 'a.id = i.album')
				->where('i.album IN (:albums)')->setParameter('albums', $albumIds)
				->andWhere('i.isCover = 1')
				->orderBy('i.position', 'ASC')
				->groupBy('a.id')
				->getQuery()->getResult();
			foreach ($imagesRaw as $image) {
				$images[$image->album->getId()] = $image;
			}

			foreach ($this->openedArticlesService->checkArticles($ids) as $v) {
				$opened[$v->article->getId()] = $v;
			}

			$this->articleHits    = $hits;
			$this->articleImages  = $images;
			$this->openedArticles = $opened;
		};

		// Columns
		$grid->addColumnText('title', 'blog.articlesGrid.title')->setSortable()->setRenderer([$this, 'gridRenderTitle']);
		$grid->addColumnText('image', 'blog.articlesGrid.image')->setRenderer([$this, 'gridRenderImage'])->setAlign('center');
		$grid->addColumnText('hits', 'blog.articlesGrid.hits')->setRenderer([$this, 'gridRenderHits'])->setAlign('right');
		$grid->addColumnText('category', 'blog.articlesGrid.category', 'category.text.title');
		$grid->addColumnStatus('featured', 'blog.articlesGrid.featured')->setAlign('center')->setTemplate(__DIR__ . '/statusColumn.latte')
			->addOption(1, 'blog.article.recommend.recommend')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(120, 'blog.article.recommend.v120')->setIcon('clock')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(300, 'blog.article.recommend.v300')->setIcon('clock')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(600, 'blog.article.recommend.v600')->setIcon('clock')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(1440, 'blog.article.recommend.v1440')->setIcon('clock')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(2880, 'blog.article.recommend.v2880')->setIcon('clock')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(4320, 'blog.article.recommend.v4320')->setIcon('clock')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(5760, 'blog.article.recommend.v5760')->setIcon('clock')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'blog.article.cancelRecommendations')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)
			->setClassInDropdown('hidden')->endOption()
			->onChange[] = [$this, 'gridFeaturedChange'];
		$grid->addColumnText('state', 'blog.articlesGrid.state', 'isPublished')->setRenderer([$this, 'gridRenderState']);
//		$grid->addColumnText('author', 'blog.articlesGrid.author', 'createdBy.name');
		$grid->addColumnDateTime('publishUp', 'blog.articlesGrid.publishUp')->setFormat('j. n. Y H:i')->setSortable();
//		$grid->addColumnDateTime('modified', 'blog.articlesGrid.modified')->setFormat('j. n. Y H:i')->setSortable();
//		$grid->addColumnText('modifiedBy', 'blog.articlesGrid.modifiedBy', 'modifiedBy.name');
		$grid->addColumnText('id', 'blog.articlesGrid.id')->setAlign('right')->setSortable();

		// Filters
		$grid->getColumn('title')->setFilterText()->setCondition(function(QueryBuilder $qb, $value) {
			$qb->andWhere('a.title LIKE :title')->setParameter('title', "%$value%");
		});

		$categoriesFilter = [];
		foreach ($categories as $k => $v)
			$categoriesFilter[$k] = $v;
		$grid->addFilterSelect('category', 'blog.articlesGrid.category', $categoriesFilter, 'a.category')->setPrompt('');

		// Sort
		$grid->getColumn('hits')->setSortable()->setSortableCallback(function(QueryBuilder $qb, $arr) {
//			if (!in_array('hit1', $qb->getAllAliases()))
//				$qb->leftJoin(Hit::class, 'hit1', Join::WITH, 'hit1.article = a.id');
			if (!in_array('hit2', $qb->getAllAliases()))
				$qb->leftJoin(Hit2::class, 'hit2', Join::WITH, 'hit2.article = a.id');
//			count(hit1.id) +
			$qb->addSelect('sum(hit2.shows) as HIDDEN hits')->orderBy('hits', $arr['hits'])->groupBy('a.id');
		});

		// Actions
		$grid->addAction('delete', '', 'delete')->setIcon('times')->addClass('ajax')->setBsType('danger')->setConfirm('blog.articlesGrid.reallyDelete');
		$grid->addGroupAction('default.deleteSelected')->onSelect[] = [$this, 'handleDelete'];

		// Columns prototype
		$grid->getColumn('id')->getElementPrototype('th')->class[] = 'w1nw';

		return $grid;
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete($id)
	{
		$presenter = $this->getPresenter();

		$result = is_array($id) ? $this->articlesService->removeArticles($id) : $this->articlesService->removeArticle($id);

		if (is_bool($result))
			if ($result)
				$presenter->flashMessageSuccess('blog.article.removed');
			else
				$presenter->flashMessageDanger('blog.article.removeFailed');
		else
			if (count($result) === 0)
				$presenter->flashMessageSuccess('blog.article.selectedRemoved');
			else
				$presenter->flashMessageInfo($this->t('blog.article.removedX', null, ['number' => count($result)]));


		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function handleInvertPublish(int $id, string $lang): void
	{
		$presenter = $this->getPresenter();

		if ($this->articlesService->invertPublish($id, $lang)) {
			$presenter->flashMessageSuccess('default.publishChanged');
		} else {
			$presenter->flashMessageDanger('default.publishChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	/**
	 * @param Article $row
	 *
	 * @throws \Nette\Application\UI\InvalidLinkException
	 */
	public function gridRenderTitle($row)
	{
		$opened = $this->openedArticles[$row->getId()] ?? null;
		if ($opened && $opened->user->getId() != $this->getPresenter()->getUser()->getId()) {
			echo Html::el('i', ['class' => 'fa fa-lock mr-2', 'data-toggle' => 'tooltip', 'data-original-title' => $this->t('blog.article.isEditingBy', null, ['user' => $opened->user->name]) . ' - ' . $opened->lastActivity->format('H:i:s')]);
			echo Html::el('span')->setText($row->title);
		} else
			echo Html::el('a', ['href' => $this->getPresenter()->link('Articles:edit', [$row->getId()])])->setText($row->getText()->title);
	}

	public function gridRenderHits($row)
	{
		return number_format($row->shows ?? 0, 0, '', ' ');
	}

	public function gridRenderImage($row)
	{
		/** @var Article $row */
		if ($row->getGallery()) {
			$image = $this->articleImages[$row->getGallery()->getId()] ?? false;
			if ($image) {
				$imageHtml = Html::el('img', ['src' => $image->getFilePath()]);
				echo Html::el('div', ['class'     => 'image-tooltip', 'data-toggle' => 'tooltip', 'data-placement' => 'right',
				                      'data-html' => 'true', 'data-original-title' => $imageHtml])->addHtml(Html::el('i', ['class' => 'fa fa-image fs-3']));
			}
		}
	}

	/**
	 * @param Article $row
	 *
	 * @return Html
	 * @throws
	 */
	public function gridRenderState($row)
	{
		$wrap = Html::el('div class=grid-langs');

		foreach ($this->langsService->getLangs(false) as $lang) {
			$text = $row->getText($lang->getTag());
			$el   = Html::el('div class=grid-langs__lang');
			if ($text) {
				$a = Html::el('a', [
					'class' => 'ajax btn btn-xs ' . ($text->isPublished ? 'btn-success' : 'btn-danger'),
					'href'  => $this->link('invertPublish!', [$row->getId(), $lang->getTag()]),
				])->setText($lang->getShortTitle());
				$el->addHtml($a);
			}
			$wrap->addHtml($el);
		}

		return $wrap;
	}

	public function gridFeaturedChange($id, $newStatus)
	{
		$presenter = $this->getPresenter();

		if ($this->articlesService->setFeature($id, $newStatus))
			$presenter->flashMessage('blog.article.featuredStatusChanged', Flashes::FLASH_SUCCESS);
		else
			$presenter->flashMessage('blog.article.featuredStatusChangeFailed', Flashes::FLASH_SUCCESS);

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function gridStatusChange($id, $newStatus)
	{
		$presenter = $this->getPresenter();

		if ($this->articlesService->setPublish($id, $newStatus))
			$presenter->flashMessageSuccess('default.publishChanged');
		else
			$presenter->flashMessageDanger('default.publishChangeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}
}
