<?php declare(strict_types = 1);

namespace Blog\AdminModule\Components\Category;

use Blog\Model\Entities\Category;
use Blog\Model\Entities\CategoryText;
use Blog\Model\UI;
use Core\Model\Lang\Langs;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Nette\Forms\Form;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use Nette\Utils\Strings;

class CategoryForm extends BaseControl
{
	/** @var Category */
	public $category;

	/** @var UI\Category\Renderer */
	protected $categoryRenderer;

	public function __construct(UI\Category\Renderer $categoryRenderer)
	{
		$this->categoryRenderer = $categoryRenderer;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm()
	{
		$form = $this->createForm();

		$layouts = [];
		foreach ($this->categoryRenderer->getLayouts() as $k => $v)
			$layouts[$k] = $v->name;

		$form->addText('title', 'blog.categoryForm.title')->setMaxLength(255)->setIsMultilanguage();
		$form->addText('alias', 'blog.categoryForm.alias')->setMaxLength(255)->setIsMultilanguage();
		$form->addSelect('layout', 'blog.categoryForm.layout', $layouts);
		$form->addText('password', 'blog.categoryForm.password');
		$form->addBool('isPublished', 'blog.categoryForm.isPublished')->setDefaultValue(1)->setIsMultilanguage();

		$form->addSubmit('submit', 'default.save');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values)
	{
		foreach ($values->title as $l => $v) {
			if (!$v) {
				$form['title']->addError('default.formMessages.required');
				break;
			}
		}

		if ($form->hasErrors()) {
			if ($form->isAjax()) {
				$this->redrawControl('form');
			} else {
				$this->redirect('this');
			}
		}
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			$langValues = $form->convertMultilangValuesToArray();
			/** @var CategoryText[] $texts */
			$texts = [];

			if ($this->category) {
				$category = $this->category;
				$texts    = $category->getTexts()->toArray();
			} else {
				$category = new Category();
			}

			foreach ($langValues as $l => $v) {
				if (!isset($texts[$l]) && $v['title'])
					$texts[$l] = new CategoryText($category, $v['title'], $l);

				if ($v['title'] == '' || $texts[$l]->getTitle() == '') {
					if ($texts[$l])
						$this->em->remove($texts[$l]);

					unset($texts[$l]);
					continue;
				}

				$texts[$l]->setTitle($v['title'])
					->setAlias($v['alias'] ?: $v['title']);
				$texts[$l]->isPublished = $v['isPublished'];

				$this->em->persist($texts[$l]);
			}

			$category->setTexts($texts);
			$category->setPassword($values->password);
			$category->setLayout($values->layout);

			$this->em->persist($category);
			$this->em->flush();
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}

		return true;
	}

	public function setCategory($id)
	{
		$this->category = $this->em->getRepository(Category::class)->find($id);

		if (!$this->category)
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);

		$form     = $this['form'];
		$category = $this->category;
		$d        = [
			'password' => $category->password,
		];

		foreach ($category->getTexts() as $l => $text) {
			$d['title'][$l]       = $text->getTitle();
			$d['alias'][$l]       = $text->getAlias();
			$d['isPublished'][$l] = $text->isPublished;
		}

		if (array_key_exists($category->layout, $form['layout']->getItems()))
			$d['layout'] = $category->layout;

		$form->setDefaults($d);
	}
}
