<?php declare(strict_types = 1);

namespace Blog\FrontModule\Model;

use Blog\Model\Entities\Article;
use Contributte\Application\LinkGenerator;
use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Images\ImagePipe;
use Core\Model\Lang\DefaultLang;
use Doctrine\ORM\Query;
use Gallery\FrontModule\Model\Albums;
use Gallery\Model\Images;
use IPub\MobileDetect\MobileDetect;
use Kdyby\Doctrine\QueryBuilder;
use Nette\Caching\Cache;
use Nette\Utils\DateTime;
use Blog\FrontModule\Model\Dao;

/**
 * Class Articles
 * @package Blog\FrontModule\Model
 *
 * @method Article|null|object = getReference($id)
 */
class Articles extends BaseFrontEntityService
{
	/** @var \stdClass */
	protected $entityClass = Article::class;

	/** @var LinkGenerator */
	public $linkGenerator;

	/** @var ImagePipe */
	protected $imagePipe;

	/** @var Albums */
	protected $albumsService;

	/** @var Categories */
	protected $categoriesService;

	const CACHE_NAMESPACE = 'articles';

	protected $cacheDep = [
		Cache::TAGS    => ['articles'],
		Cache::EXPIRE  => '1 month',
		Cache::SLIDING => true,
	];

	public function __construct(ImagePipe $imagePipe, Albums $albums, Categories $categories)
	{
		$this->imagePipe         = $imagePipe;
		$this->albumsService     = $albums;
		$this->categoriesService = $categories;
	}

	/**
	 * @return Cache|\Nette\Caching\Cache
	 */
	protected function getCache()
	{
		if ($this->cache === null)
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);

		return $this->cache;
	}

	/**
	 * @param QueryBuilder $qb
	 * @param bool         $withLockedCategory
	 *
	 * @return QueryBuilder
	 * @throws \Exception
	 */
	public function setPublishedCriteria(QueryBuilder $qb, $withLockedCategory = false)
	{
		$now = (new DateTime())->format('Y-m-d H:i:00');
		$qb->andWhere('a.publishUp IS NULL OR a.publishUp < :now')->andWhere('a.publishDown IS NULL OR a.publishDown > :now')
			->innerJoin('a.texts', 'at', 'WITH', 'at.lang = :lang AND at.isPublished = 1')
			->setParameter('lang', $this->translator->getLocale())
			->setParameter('now', $now);

		if (!$withLockedCategory) {
			$qb->andWhere('c.password IS NULL');
		}

		return $qb;
	}

	public function getForNavigation(int $id, string $lang)
	{
		return $this->getEr()->createQueryBuilder('a')->select('a.id, at.alias, c.id as category, ct.alias as categoryAlias')
			->innerJoin('a.texts', 'at', 'WITH', 'at.lang = :lang AND at.isPublished = 1')
			->innerJoin('a.category', 'c')
			->innerJoin('c.texts', 'ct', 'WITH', 'ct.lang = :lang')
			->setParameter('lang', $lang)
			->andWhere('a.id = :id')->setParameter('id', $id)
			->getQuery()->getOneOrNullResult(Query::HYDRATE_ARRAY);
	}


	public function get(int $id)
	{
		$key = 'article/' . $id . '/' . $this->translator->getLocale();
		/** @var Dao\Article $dao */
		$dao = $this->getCache()->load($key, function(&$dep) use ($id) {
			$dep = $this->cacheDep;

			$row = $this->getEr()->createQueryBuilder('a')
				->addSelect('at, IDENTITY(a.gallery) as gallery, IDENTITY(a.category) as category')
				->innerJoin('a.texts', 'at', 'WITH', 'at.lang = :lang AND at.isPublished = 1')
				->andWhere('a.id = :id')
				->setParameters([
					'lang' => $this->translator->getLocale(),
					'id'   => $id,
				])->getQuery()->getOneOrNullResult(Query::HYDRATE_ARRAY);

			if (!$row)
				return null;

			$article = $row[0];
			$text    = $article['texts'][$this->translator->getLocale()];

			$dao              = new Dao\Article();
			$dao->id          = (int) $article['id'];
			$dao->lang        = $this->translator->getLocale();
			$dao->title       = $text['title'];
			$dao->alias       = $text['alias'];
			$dao->introtext   = $text['introtext'];
			$dao->fulltext    = $text['fulltext'];
			$dao->created     = $text['created'];
			$dao->modified    = $text['modified'];
			$dao->featured    = $article['featured'];
			$dao->params      = $article['params'];
			$dao->categoryId  = (int) $row['category'];
			$dao->galleryId   = (int) $row['gallery'];
			$dao->publishUp   = $article['publishUp'];
			$dao->publishDown = $article['publishDown'];
			$dao->seo         = $text['seo'];

			return $dao;
		});

		$dao->link        = $this->linkGenerator->link('Blog:Front:Articles:detail', ['id' => $dao->getId()]);
		$dao->category = $this->categoriesService->get($dao->categoryId);
		$dao->gallery  = $this->albumsService->get($dao->galleryId);

		return $dao;
	}

	public function getPublishedIdByAlias($alias, $lang)
	{
		$now = (new DateTime())->format('Y-m-d H:i:00');

		return $this->getEr()->createQueryBuilder('a')
			->select('a.id')
			->innerJoin('a.texts', 'at', 'WITH', 'at.lang = :lang AND at.isPublished = 1 AND at.alias = :alias')
			->setParameter('lang', $lang)
			->setParameter('alias', $alias)
			->andWhere('a.publishUP IS NULL OR a.publishUp < :now')->andWhere('a.publishDown IS NULL OR a.publishDown > :now')
			->setMaxResults(1)
			->getQuery()->getSingleScalarResult();
	}

	/**
	 * @param null   $categories
	 * @param null   $offset
	 * @param null   $limit
	 * @param string $sort
	 * @param bool   $withLockedCategories
	 *
	 * @return array
	 * @throws \Exception
	 */
	public function getAll($categories = null, $offset = null, $limit = null, $sort = 'publishUp', $withLockedCategories = false)
	{
		$now      = (new DateTime())->format('Y-m-d H:i:00');
		$articles = [];

		$qb = $this->getEr()->createQueryBuilder('a')
			->select('a.id')
			->innerJoin('a.texts', 'at', 'WITH', 'at.lang = :lang AND at.isPublished = 1')
			->setParameter('lang', $this->translator->getLocale())
			->andWhere('a.publishUp < :now')->andWhere('a.publishDown IS NULL OR a.publishDown < :now')
			->setParameter('now', $now);

		if (is_array($categories) && count($categories) == 1)
			$categories = $categories[0];
		if ($categories) {
			$qb->andWhere(is_numeric($categories) ? 'a.category = :catId' : 'a.category IN (:catId)')
				->setParameter('catId', $categories);
		}

		if ($offset)
			$qb->setFirstResult($offset);
		if ($limit)
			$qb->setMaxResults($limit);

		if (is_string($sort)) {
			switch ($sort) {
				case 'default':
				case 'publishUp':
					$qb->orderBy('a.publishUp', 'DESC');
					break;
			};
		} else if (is_array($sort)) {
			foreach ($sort as $k => $v) {
				$qb->addOrderBy($k, $v);
			}
		}

		$ids = $qb->getQuery()->getScalarResult();

		foreach ($ids as $row) {
			$a = $this->get((int) $row['id']);
			if ($a)
				$articles[$row['id']] = $a;
		}

		return $articles;
	}

	/**
	 * @param null $categories
	 * @param bool $withLockedCategories
	 *
	 * @return int
	 * @throws \Exception
	 */
	public function getAllCount($categories = null, $withLockedCategories = false)
	{
		return count($this->getAll($categories, null, null, null, $withLockedCategories));
	}

	/**
	 * @param Article $article
	 * @param int     $limit
	 * @param bool    $withLockedCategories
	 *
	 * @return array
	 * @throws \Exception
	 */
	public function getNextPublishedArticle(Article $article, $limit = 1, $withLockedCategories = false)
	{
		$articles = [];

		$qb = $this->getEr()->createQueryBuilder('a');
		$qb = $this->setPublishedCriteria($qb, $withLockedCategories);

		$qb->andWhere('a.publishUp > :articlePublishUp')->setParameter('articlePublishUp', $article->publishUp)
			->orderBy('a.publishUp', 'ASC')
			->select('a.id')
			->setMaxResults($limit);

		foreach ($qb->getQuery()->getScalarResult() as $row) {
			$a = $this->get($row['id']);
			if ($a)
				$articles[$row['id']] = $a;
		}

		return $articles;
	}

	/**
	 * @param Dao\Article $article
	 * @param int         $limit
	 * @param bool        $withLockedCategories
	 *
	 * @return array
	 * @throws \Exception
	 */
	public function getPrevPublishedArticle(Dao\Article $article, $limit = 1, $withLockedCategories = false)
	{
		$articles = [];

		$qb = $this->getEr()->createQueryBuilder('a');
		$qb = $this->setPublishedCriteria($qb, $withLockedCategories);

		$qb->andWhere('a.publishUp < :articlePublishUp')->setParameter('articlePublishUp', $article->publishUp)
			->orderBy('a.publishUp', 'DESC')
			->select('a.id')
			->setMaxResults($limit);

		foreach ($qb->getQuery()->getScalarResult() as $row) {
			$a = $this->get($row['id']);
			if ($a)
				$articles[$row['id']] = $a;
		}

		return $articles;
	}

	/**
	 * Přidání zobrazení ke článku
	 *
	 * @param int $articleId
	 */
	public function addShow(int $articleId)
	{
		$this->getEr()->createQuery('UPDATE ' . $this->entityClass . ' a SET a.shows = a.shows + 1 WHERE a.id = :id')
			->setParameters([':id' => $articleId])->execute();
	}
}
