<?php declare(strict_types = 1);

namespace Blog\Model;

use Blog\Model\Entities\Article;
use Blog\Model\Entities\Opened;
use Core\Model\Helpers\BaseEntityService;
use Nette\Utils\DateTime;
use Users\Model\Entities\User;

class OpenedArticlesService extends BaseEntityService
{

	/** @var \stdClass */
	protected $entityClass = Opened::class;

	/**
	 * @param int $id
	 *
	 * @return Opened|null|object
	 */
	public function isArticleOpened($id)
	{
		return $this->getEr()->findOneBy(['article' => $id]);
	}

	/**
	 * @param array $ids
	 *
	 * @return Opened[]|null
	 */
	public function isArticlesOpened($ids)
	{
		return $this->getEr()->findBy(['article' => $ids]);
	}

	/**
	 * @param Article $article
	 * @param User    $user
	 *
	 * @throws \Exception
	 */
	public function openArticle($article, $user)
	{
		$opened = $this->isArticleOpened($article->getId());
		if (!$opened)
			$opened = new Opened($article, $user);

		$opened->user         = $user;
		$opened->lastActivity = new DateTime();

		$this->em->persist($opened);
		$this->em->flush();
	}

	/**
	 * Zkontroluje aktivitu a vrátí aktivní článek
	 *
	 * @param int $id
	 *
	 * @return Opened|null
	 * @throws \Exception
	 */
	public function checkArticle($id)
	{
		$date = (new DateTime())->modify('-5 minutes');

		$opened = $this->isArticleOpened($id);
		if ($opened && $opened->lastActivity < $date) {
			$this->em->remove($opened);
			$this->em->flush();
		}

		return $opened;
	}

	/**
	 * Zkontroluje poslední aktivitu a vrátí pole otevřených článků
	 *
	 * @param $ids
	 *
	 * @return Opened[]|null
	 * @throws \Exception
	 */
	public function checkArticles($ids)
	{
		$date = (new DateTime())->modify('-5 minutes');

		$opened = $this->isArticlesOpened($ids);
		foreach ($opened as $open) {
			if ($open && $open->lastActivity < $date)
				$this->em->remove($open);
		}
		if ($opened)
			$this->em->flush();

		return $opened;
	}

	public function closeArticle($ids)
	{
		$opened = $this->isArticlesOpened($ids);
		foreach ($opened as $open) {
			if ($open)
				$this->em->remove($open);
		}
		if ($opened)
			$this->em->flush();
	}

}