<?php declare(strict_types = 1);

namespace Blog\AdminModule\Components\Category;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Parameter;
use Blog\Model\Articles;
use Blog\Model\Categories;
use Blog\Model\Entities\Category;
use Blog\Model\Entities\CategoryText;
use Core\Model\Helpers\Arrays;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Nette\Application\UI\Multiplier;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;

class CategoriesGrid extends BaseControl
{
	protected ?array $cData = null;

	public function __construct(
		protected string               $siteIdent,
		protected Categories           $categoriesService,
		protected Articles             $articlesService,
		protected ICategoryFormFactory $categoryFormFactory,
	)
	{
	}

	public function render(): void
	{
		$this->template->siteIdent  = $this->siteIdent;
		$this->template->categories = $this->getData()['tree'];
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete(int $id): void
	{
		$presenter = $this->presenter;
		if ($this->categoriesService->removeCategory($id)) {
			$presenter->flashMessageSuccess('default.removed');
		} else {
			$presenter->flashMessageDanger('default.removeFailed');
		}

		$this->redrawControl('list');
		$presenter->redrawControl('flashes');
	}

	public function handleEdit(int $id, string $siteIdent): void
	{
		$category = $this->categoriesService->get($id);

		if (!$category) {
			throw new InvalidArgumentException;
		}

		$this['categoryForm']->categoryId = $id;
		$this['categoryForm']->siteIdent  = $siteIdent;
		$this->template->modalTitle       = $this->t('blog.title.editCategory') . ' ' . $category->getText()
				->getTitle();
		$this->template->modal            = 'categoryForm';
		$this->redrawControl('modal');
	}

	public function handleChangePosition(): void
	{
		$presenter = $this->presenter;
		$request   = $presenter->getHttpRequest();

		$id   = (int) $request->getPost('id');
		$move = (int) $request->getPost('move');

		if ($id && $move) {
			$er  = $this->categoriesService->getEr();
			$cat = $this->categoriesService->get($id);

			if ($cat) {
				if ($move < 0) {
					$er->moveUp($cat, abs($move));
				} else {
					$er->moveDown($cat, $move);
				}
				$presenter->flashMessageSuccess('default.positionChanged');
			}
		} else {
			$presenter->flashMessageDanger('default.positionChangeFailed');
		}

		$this->em->flush();

		$presenter->redrawControl('flashes');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentCategoryForm(): CategoryForm
	{
		$control = $this->categoryFormFactory->create();
		$params  = $this->getParameters();

		if ($params['siteIdent']) {
			$control->siteIdent = $params['siteIdent'];
		}

		if ($params['id']) {
			$control->setCategory((int) $params['id']);
		}

		$control['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this->redrawControl('list');
			$this->presenter->redrawControl('flashes');
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->redrawControl('list');
			$this->presenter->payload->hideModal = true;
			$this->presenter->redrawControl('flashes');
		};
		$control['form']->getComponent('saveControl')->closeModalOnCancel();

		return $control;
	}

	protected function createComponentPublishForm(): Multiplier
	{
		return new Multiplier(function($id): BaseForm {
			$form = $this->createForm();

			$form->addBool('publish', '')
				->setDefaultValue($this->getData()['data'][$id]['isPublished'] ?? 0)
				->setHtmlAttribute('onchange', 'window.naja.uiHandler.submitForm(this.form)');

			$form->onSuccess[] = function(BaseForm $form, ArrayHash $values) use ($id) {
				$presenter = $this->presenter;

				if ($this->categoriesService->setPublish((int) $id, $values->publish)) {
					$presenter->flashMessageSuccess('default.publishChanged');
				} else {
					$presenter->flashMessageDanger('default.publishChangeFailed');
				}

				$presenter->redrawControl('flashes');
			};

			return $form;
		});
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function getData(): array
	{
		if ($this->cData !== null) {
			return $this->cData;
		}

		$lang = $this->translator->getLocale();

		$rootId = $this->categoriesService->getEr()->createQueryBuilder('c')
			->select('c.id')
			->innerJoin('c.texts', 'ct')
			->where('c.lvl = :lvl')
			->andWhere('ct.alias = :alias')
			->setParameters(new ArrayCollection([
				new Parameter('lvl', 0),
				new Parameter('alias', $this->siteIdent),
			]))->getQuery()->setMaxResults(1)->getArrayResult()[0]['id'] ?? null;

		if (!$rootId) {
			$newCategory     = new Category();
			$newCategoryText = new CategoryText($newCategory, '', $lang);
			$newCategoryText->setAlias($this->siteIdent);

			$this->em->persist($newCategory);
			$this->em->persist($newCategoryText);
			$this->em->flush();

			$rootId = $newCategory->getId();

			/** @var Category[] $tmp */
			$tmp = $this->categoriesService->getEr()->findBy(['parent' => null, 'root' => null]);

			if (!empty($tmp)) {
				foreach ($tmp as $row) {
					$row->parent = $newCategory;
					$this->em->persist($row);
				}

				$this->em->flush();
			}
		}

		$rootTexts = [];
		foreach ($this->em->getConnection()->iterateAssociative("SELECT category_id, lang FROM blog__category_text WHERE alias = :alias", [
			'alias' => $this->siteIdent,
		]) as $row) {
			/** @var array $row */
			$rootTexts[$row['lang']] = $row['category_id'];
		}

		foreach ($this->langsService->getLangs(false) as $l) {
			if (!isset($rootTexts[$l->getTag()])) {
				$newCategoryText = new CategoryText($this->em->getReference(Category::class, $rootId), '', $l->getTag());
				$newCategoryText->setAlias($this->siteIdent);

				$this->em->persist($newCategoryText);
				$this->em->flush();
			}
		}

		// Kategorie
		$qb   = $this->categoriesService->getEr()->createQueryBuilder('c', 'c.id')
			->select('c.id, c.isPublished, ct.title, IDENTITY(c.parent) as parent')
			->leftJoin('c.texts', 'ct')
			->where('ct.lang = :lang')
			->andWhere('c.root = :root')
			->setParameters(new ArrayCollection([
				new Parameter('lang', $lang),
				new Parameter('root', $rootId),
			]))
			->groupBy('c.id')
			->addOrderBy('c.lft');
		$data = $qb->getQuery()->getArrayResult();

		$this->cData = [
			'data' => $data,
			'tree' => Arrays::buildTree($data, 'parent', 'id', $rootId),
		];

		return $this->cData;
	}

}
