<?php declare(strict_types = 1);

namespace Blog\AdminModule\Model;

use Blog\Model\Entities\Category;
use Blog\Model\Entities\CategoryText;
use Contributte\Translation\Translator;
use Core\Model\Entities\Repository\NestedTreeRepository;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Helpers\Traits\TPublish;

/**
 * @method Category|null getReference($id)
 * @method Category[] getAll()
 * @method NestedTreeRepository getEr()
 */
class Categories extends BaseEntityService
{
	use TPublish;

	protected       $entityClass = Category::class;
	protected array $cForTree    = [];

	public function __construct(
		protected Translator $translator,
	)
	{
	}

	protected function prepareDataForTree(?string $siteIdent = null): array
	{
		$cKey = $siteIdent ?: 'default';
		if ($this->cForTree[$cKey] === null) {
			$rootId = null;

			if ($siteIdent) {
				$rootId = $this->em->getRepository(CategoryText::class)->createQueryBuilder('ct')
					->select('IDENTITY(ct.category) as id')
					->where('ct.alias = :alias')->setParameter('alias', $siteIdent)
					->getQuery()->setMaxResults(1)->getArrayResult()[0]['id'] ?? null;
			}
			$roots      = [];
			$categories = [];
			$flat       = [];

			if (($siteIdent && $rootId) || !$siteIdent) {
				$qb = $this->getEr()->createQueryBuilder('c')
					->select('c.id, ct.title, c.isPublished, c.lvl, IDENTITY(c.parent) as parent')
					->leftJoin('c.texts', 'ct', 'WITH', 'ct.lang = :lang')
					->setParameter('lang', $this->translator->getLocale())
					->groupBy('c.id')
					->addOrderBy('c.root')->addOrderBy('c.lft');

				if ($rootId) {
					$qb->andWhere('c.root = :root')->setParameter('root', $rootId);
				}

				foreach ($qb->getQuery()->getResult() as $c) {
					if ($c['lvl'] == 0) {
						$roots[$c['id']] = [
							'id'   => $c['id'],
							'name' => $c['title'],
						];
						continue;
					}

					$name = !$c['isPublished'] ? '[x ' . $c['title'] . ' x]' : $c['title'];

					$categories[$c['id']] = $name;
					$flat[]               = [
						'id'     => $c['id'],
						'parent' => $c['parent'],
						'name'   => $name,
					];
				}
			}

			$this->cForTree[$cKey] = [
				'roots'      => $roots,
				'categories' => $categories,
				'flat'       => $flat,
			];
		}

		return $this->cForTree[$cKey];
	}

	public function getOptionsForSelect(?string $siteIdent = null): array
	{
		return $this->prepareDataForTree($siteIdent)['categories'];
	}

	public function getFlatTree(?string $siteIdent = null): array
	{
		return $this->prepareDataForTree($siteIdent)['flat'];
	}

	public function getTreeForSelect(?string $siteIdent = null): array
	{
		$data = $this->prepareDataForTree($siteIdent);

		$trees = [];
		foreach ($data['roots'] as $id => $root) {
			$trees[$id] = [
				'name'     => $root['title'],
				'children' => Arrays::buildTree($data['flat'], 'parent', 'id', $root['id']),
			];
		}

		return $trees;
	}
}
