<?php declare(strict_types = 1);

namespace Blog\FrontModule\Model;

use Blog\FrontModule\Model\Dao\Author;
use Blog\Model\BlogCache;
use Blog\Model\BlogConfig;
use Blog\Model\Entities\Article;
use Blog\Model\Entities\CategoryText;
use Core\Model\Entities\ExtraField;
use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Images\ImagePipe;
use Core\Model\Sites;
use DateTimeInterface;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping\ClassMetadataInfo;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\Query\Parameter;
use Doctrine\ORM\QueryBuilder;
use Exception;
use Gallery\FrontModule\Model\Albums;
use Navigations\Model\Entities\Navigation;
use Nette\Application\LinkGenerator;
use Nette\Application\UI\InvalidLinkException;
use Nette\Caching\Cache;
use Nette\Utils\DateTime;
use Tags\FrontModule\Model\Tags;
use Throwable;
use Users\Model\Entities\User;

/**
 * @method Article getReference($id)
 */
class Articles extends BaseFrontEntityService
{
	protected $entityClass = Article::class;

	public ?LinkGenerator $linkGenerator = null;

	protected array $cArticles = [], $cNavs = [], $cAuthors = [];

	protected ?array $cRoot = null;

	protected array $cForNav = [];

	protected ?array $cTags = null;

	public const CACHE_NAMESPACE = 'articles';

	public array $cacheDep = [
		Cache::Tags   => ['articles'],
		Cache::Expire => '1 week',
	];

	public function __construct(
		protected ImagePipe  $imagePipe,
		protected Albums     $albumsService,
		protected Categories $categoriesService,
		protected Sites      $sitesService,
		protected BlogCache  $blogCache,
		protected Tags       $tagsService
	)
	{
	}

	/**
	 * TODO replace with Blog\Model\BlogCache
	 */
	protected function getCache(): Cache
	{
		return $this->blogCache->getCache();
	}

	/**
	 * @param string[]|int[] $ids
	 *
	 * @return Dao\Article[]
	 * @throws Throwable
	 * @throws InvalidLinkException
	 */
	public function getArticles(array $ids): array
	{
		$sort     = $ids;
		$whereIds = [];
		/** @var Dao\Article[] $result */
		$result = [];
		/** @var Dao\Article[] $loaded */
		$loaded = [];
		$locale = $this->translator->getLocale();

		foreach ($ids as $k => $id) {
			if (isset($this->cArticles[$id])) {
				$loaded[$id] = $this->cArticles[$id];
				unset($ids[$k]);
			}
		}

		$keys = [];
		foreach ($ids as $id) {
			$keys[] = 'article/' . $id . '/' . $locale;
		}

		// Načtení z cache
		if ($keys) {
			foreach ($this->getCache()->bulkLoad($keys) as $key => $article) {
				$tmp = explode('/', $key);
				$id  = $tmp[1];

				if ($article) {
					$result[$id] = $article;
				} else {
					$whereIds[] = $id;
				}
			}
		}

		// Vytvoření získání clanku co nejsou v cache
		if (!empty($whereIds)) {
			$rootCat = $this->getCategoryRootForSite($this->sitesService->getCurrentSite()->getIdent());

			$qb = $this->getEr()->createQueryBuilder('a')
				->addSelect('at, IDENTITY(a.gallery) as gallery, IDENTITY(at.createdBy) as createdBy')
				->innerJoin('a.texts', 'at', 'WITH', 'at.lang = :lang AND at.isPublished = 1')
				->setParameter('lang', $this->translator->getLocale());

			if (BlogConfig::load('allowTags')) {
				$qb->addSelect('GROUP_CONCAT(tags.id) as tagIds')
					->leftJoin('a.tags', 'tags');
			}

			if (count($whereIds) === 1) {
				$qb->andWhere('a.id = :id');
			} else {
				$qb->andWhere('a.id IN (:id)');
			}
			$qb->setParameter('id', $whereIds);

			if ($rootCat) {
				$qb->innerJoin('a.categories', 'aic')
					->innerJoin('aic.category', 'cat', Join::WITH, 'cat.root = :root')
					->setParameter('root', $rootCat)
					->setParameter('lang', $this->translator->getLocale())
					->addSelect('cat.id as catId');
			}

			if (BlogConfig::load('allowTags')) {
				$qb->addGroupBy('a.id')->addGroupBy('at.lang');
			}

			$tmp = [];
			foreach ($qb->getQuery()->getArrayResult() as $row) {
				$arr                = $this->normalizeHydrateArray($row);
				$arr['tagIds']      = $row['tagIds'];
				$tmp[$row[0]['id']] = $arr;
			}

			foreach ($this->em->getRepository(ExtraField::class)->createQueryBuilder('ef')
				         ->select('ef.sectionKey, ef.key, ef.value')
				         ->andWhere('ef.sectionName = :name')
				         ->andWhere('ef.sectionKey IN (:keys)')
				         ->andWhere('ef.lang = :lang OR ef.lang IS NULL')
				         ->setParameters(new ArrayCollection([new Parameter('name', Article::EXTRA_FIELD_SECTION), new Parameter('keys', array_keys($tmp)),
					         new Parameter('lang', $this->translator->getLocale())]))->getQuery()->getArrayResult() as $row) {
				$tmp[$row['sectionKey']]['params'][$row['key']] = $row['value'];
			}

			foreach ($tmp as $row) {
				$dao                     = $this->fillDao($row);
				$cacheDep                = $this->cacheDep;
				$cacheDep[Cache::Tags][] = 'article/' . $dao->getId();
				$this->getCache()->save('article/' . $dao->getId() . '/' . $locale, $dao, $cacheDep);
				$result[$row['id']] = $dao;
			}
		}

		if (!empty($result)) {
			// Preload
			$categories = [];
			$albums     = [];
			foreach ($result as $article) {
				if ($article->categoryId) {
					$categories[] = $article->categoryId;
				}
				if ($article->galleryId) {
					$albums[] = $article->galleryId;
				}
			}

			if ($categories) {
				$categories = $this->categoriesService->getCategories($categories);
			}
			if ($albums) {
				$albums = $this->albumsService->getAlbums($albums);
			}

			foreach ($result as $id => $article) {
				$article->link = $this->linkGenerator->link('Blog:Front:Articles:detail', [
					'id'     => $article->getId(),
					'locale' => $locale,
				]);

				if ($article->categoryId) {
					$article->category = $categories[$article->categoryId] ?? null;
				}
				if ($article->galleryId) {
					$article->gallery = $albums[$article->galleryId] ?? null;
				}

				$this->cArticles[$id] = $article;
				$loaded[$id]          = $article;
				$result[$id]          = null;
			}

			$categories = null;
			$albums     = null;
		}

		$authorKeys     = [];
		$authors        = [];
		$authorWhereIds = [];

		foreach ($loaded as $k => $v) {
			if ($v->createdById) {
				if (isset($this->cAuthors[$v->createdById])) {
					$loaded[$k]->createdBy = $this->cAuthors[$v->createdById];
				} else {
					$authorKeys[] = 'author_' . $v->createdById;
				}
			}
		}

		if (!empty($authorKeys)) {
			foreach ($this->blogCache->getCache()->bulkLoad($authorKeys) as $key => $author) {
				$tmp = explode('_', $key);
				$id  = $tmp[1];

				if ($author) {
					$authors[$id]        = $author;
					$this->cAuthors[$id] = $author;
				} else {
					$authorWhereIds[]    = $id;
					$this->cAuthors[$id] = null;
				}
			}
		}

		if (!empty($authorWhereIds)) {
			foreach (array_chunk($authorWhereIds, 900) as $chunk) {
				foreach ($this->em->getRepository(User::class)->createQueryBuilder('u')
					         ->select('u.id, u.name, u.lastname')
					         ->where('u.id IN (:ids)')
					         ->setParameter('ids', $chunk)
					         ->getQuery()->getArrayResult() as $row) {
					$authors[$row['id']]        = new Author((int) $row['id'], $row['name'], $row['lastname']);
					$this->cAuthors[$row['id']] = $authors[$row['id']];
					$this->blogCache->getCache()->save('author_' . $row['id'], $authors[$row['id']], [
						Cache::Tags   => 'author',
						Cache::Expire => '1 week',
					]);
				}
			}
		}

		if (!empty($authors)) {
			foreach ($loaded as $k => $v) {
				if ($v->createdById && !$v->createdBy && isset($authors[$v->createdById])) {
					$loaded[$k]->createdBy = $authors[$v->createdById];
				}
			}
		}

		$sorted = [];
		foreach ($sort as $id) {
			if (isset($loaded[$id])) {
				$sorted[$id] = $loaded[$id];
			}
		}

		return $sorted;
	}

	public function setPublishedCriteria(
		QueryBuilder $qb,
		bool         $withLockedCategory = false,
	): QueryBuilder
	{
		$now = (new DateTime())->format('Y-m-d H:i:00');
		$qb->andWhere('a.publishUp IS NULL OR a.publishUp < :now')
			->andWhere('a.publishDown IS NULL OR a.publishDown >= :now')
			->innerJoin('a.texts', 'at', 'WITH', 'at.lang = :lang AND at.isPublished = 1')
			->setParameter('lang', $this->translator->getLocale())
			->setParameter('now', $now);

		if (!$withLockedCategory && in_array('c', $qb->getAllAliases(), true)) {
			$qb->innerJoin('aic.category', 'c');
			$qb->andWhere('c.password IS NULL');
		}

		return $qb;
	}

	public function getForNavigation(int $id, string $lang): ?array
	{
		$cRoot = $this->getCategoryRootForSite($this->sitesService->getCurrentSite()->getIdent());

		if (!$cRoot) {
			return null;
		}

		if (!isset($this->cForNav[$cRoot][$lang])) {
			$this->cForNav[$cRoot][$lang] = $this->getCache()->load('forNavigation/' . $cRoot . '/' . $lang, function(
				&$dep,
			) use ($cRoot, $lang) {
				$dep                = $this->cacheDep;
				$dep[Cache::Tags][] = 'articleNavs';
				$arr                = [];

				foreach ($this->getEr()
					         ->createQueryBuilder('a')
					         ->select('a.id, at.alias, c.id as category, ct.alias as categoryAlias')
					         ->innerJoin('a.texts', 'at', 'WITH', 'at.lang = :lang AND at.isPublished = 1')
					         ->innerJoin('a.categories', 'cats')
					         ->innerJoin('cats.category', 'c', Join::WITH, 'c.root = :cRoot')
					         ->innerJoin('c.texts', 'ct', 'WITH', 'ct.lang = :lang')
					         ->setParameters(new ArrayCollection([new Parameter('cRoot', $cRoot), new Parameter('lang', $lang)]))
					         ->getQuery()
					         ->getArrayResult() as $row) {
					$arr[$row['id']] = $row;
				}

				return $arr;
			});
		}

		return $this->cForNav[$cRoot][$lang][$id];
	}

	public function get(int $id): ?Dao\Article
	{
		return $this->getArticles([$id])[$id] ?? null;
	}

	public function getPublishedIdByAlias(string $alias, string $lang): ?array
	{
		$now = (new DateTime)->format('Y-m-d H:i:00');

		return $this->getEr()
			->createQueryBuilder('a')
			->select('a.id')
			->innerJoin('a.texts', 'at', 'WITH', 'at.lang = :lang AND at.isPublished = 1 AND at.alias = :alias')
			->andWhere('a.publishUp IS NULL OR a.publishUp < :now')
			->andWhere('a.publishDown IS NULL OR a.publishDown >= :now')
			->setParameters(new ArrayCollection([new Parameter('lang', $lang), new Parameter('alias', $alias), new Parameter('now', $now)]))
			->setMaxResults(1)
			->getQuery()
			->getArrayResult()[0] ?? null;
	}

	/**
	 * @param string[]|int[]|null  $categories
	 * @param string[]|string|null $sort
	 *
	 * @return Dao\Article[]
	 * @throws Throwable
	 */
	public function getAll(
		?array            $categories = null,
		string|int|null   $offset = null,
		string|int|null   $limit = null,
		array|string|null $sort = 'publishUp',
		bool              $withLockedCategories = false,
	): array
	{
		$siteIdent = $this->sitesService->getCurrentSite()->getIdent();
		$rootId    = $this->getCategoryRootForSite($siteIdent);
		$cacheKey  = 'getAll_' . $siteIdent . '-' . $rootId . '_' . $this->translator->getLocale() . '_'
			. md5(serialize([$categories, $sort, $withLockedCategories]));

		$ids = $this->blogCache->getCache()->load($cacheKey, function(&$dep) use ($categories, $sort) {
			$dep = [
				Cache::Tags   => [BlogCache::cacheNamespace, 'getAll'],
				Cache::Expire => '5 minutes',
			];

			$now = (new DateTime)->format('Y-m-d H:i:00');

			$qb = $this->getEr()->createQueryBuilder('a')
				->select('a.id')
				->innerJoin('a.texts', 'at', 'WITH', 'at.lang = :lang AND at.isPublished = 1')
				->setParameter('lang', $this->translator->getLocale())
				->andWhere('a.publishUp < :now')->andWhere('a.publishDown IS NULL OR a.publishDown >= :now')
				->setParameter('now', $now);

			if (is_array($categories) && count($categories) === 1) {
				$categories = $categories[0];
			}

			$qb->innerJoin('a.categories', 'aCat');
			if ($categories) {
				$qb->andWhere(is_numeric($categories) ? 'aCat.category = :catId' : 'aCat.category IN (:catId)')
					->setParameter('catId', $categories);
			}

			$root = $this->categoriesService->getRootCategory();
			if ($root) {
				$qb->innerJoin('aCat.category', 'cats', Join::WITH, 'cats.root = :root')
					->setParameter('root', $root->getId());
			}

			if (is_string($sort)) {
				switch ($sort) {
					case 'default':
					case 'publishUp':
						$qb->orderBy('a.publishUp', 'DESC');
						break;
				}
			} else if (is_array($sort)) {
				foreach ($sort as $k => $v) {
					$qb->addOrderBy($k, $v);
				}
			}

			$ids = $qb->getQuery()->getScalarResult();

			return array_map(static fn($v) => $v['id'], $ids);
		});

		if ($limit > 0) {
			$ids = array_slice($ids, (int) $offset, (int) $limit);
		}

		return $this->getArticles($ids);
	}

	/**
	 *
	 * @throws Throwable
	 */
	public function getAllCount(?array $categories = null, bool $withLockedCategories = false): int
	{
		return count($this->getAll($categories, null, null, null, $withLockedCategories));
	}

	/**
	 * @return Dao\Article[]
	 * @throws Throwable
	 */
	public function getFeatured(int $limit = 8): array
	{
		$ids = $this->blogCache->getCache()->load('featured:' . $this->translator->getLocale(), function(&$dep) {
			$dep = [
				Cache::Tags   => [BlogCache::cacheNamespace, 'featured'],
				Cache::Expire => '5 minutes',
			];

			$qb = $this->getEr()->createQueryBuilder('a');
			$qb = $this->setPublishedCriteria($qb);

			$qb->select('a.id')
				->orderBy('a.publishUp', 'DESC')
				->andWhere('a.featured >= :now OR a.featured = \'1\'')
				->setParameter('now', (new DateTime));

			$arr = [];
			foreach ($qb->getQuery()->getScalarResult() as $row) {
				$arr[] = $row['id'];
			}

			return $arr;
		});

		$ids = array_slice($ids, 0, $limit);

		return $this->getArticles($ids);
	}

	/**
	 * @return Dao\Article[]
	 * @throws Exception
	 * @deprecated getNextPublishedArticles
	 */
	public function getNextPublishedArticle(
		Dao\Article $article,
		int         $limit = 1,
		bool        $withLockedCategories = false,
	): array
	{
		$articles = [];

		$qb = $this->getEr()->createQueryBuilder('a');
		$qb = $this->setPublishedCriteria($qb, $withLockedCategories);

		$qb->andWhere('a.publishUp > :articlePublishUp')->setParameter('articlePublishUp', $article->publishUp)
			->orderBy('a.publishUp', 'ASC')
			->select('a.id')
			->setMaxResults($limit);

		foreach ($qb->getQuery()->getScalarResult() as $row) {
			$a = $this->get($row['id']);
			if ($a) {
				$articles[$row['id']] = $a;
			}
		}

		return $articles;
	}

	public function getNextPublishedArticles(
		DateTimeInterface $publishUp,
		array             $cats,
		int               $limit = 1,
		bool              $withLockedCategories = false,
	): array
	{
		$articles = [];

		$qb = $this->getEr()->createQueryBuilder('a')
			->innerJoin('a.categories', 'aic');
		$qb = $this->setPublishedCriteria($qb, $withLockedCategories);

		$qb->andWhere('a.publishUp > :publishUp')->setParameter('publishUp', $publishUp)
			->andWhere('aic.category IN (:cats)')->setParameter('cats', $cats)
			->orderBy('a.publishUp', 'ASC')
			->select('a.id')
			->setMaxResults($limit);

		foreach ($qb->getQuery()->getArrayResult() as $row) {
			$a = $this->get($row['id']);
			if ($a) {
				$articles[$row['id']] = $a;
			}
		}

		return $articles;
	}

	/**
	 * @return Dao\Article[]
	 * @throws Exception
	 * @deprecated getPrevPublishedArticles
	 */
	public function getPrevPublishedArticle(
		Dao\Article $article,
		int         $limit = 1,
		bool        $withLockedCategories = false,
	): array
	{
		$articles = [];

		$qb = $this->getEr()->createQueryBuilder('a');
		$qb = $this->setPublishedCriteria($qb, $withLockedCategories);

		$qb->andWhere('a.publishUp < :articlePublishUp')->setParameter('articlePublishUp', $article->publishUp)
			->orderBy('a.publishUp', 'DESC')
			->select('a.id')
			->setMaxResults($limit);

		foreach ($qb->getQuery()->getScalarResult() as $row) {
			$a = $this->get($row['id']);
			if ($a) {
				$articles[$row['id']] = $a;
			}
		}

		return $articles;
	}

	/**
	 * @return Dao\Article[]
	 */
	public function getPrevPublishedArticles(
		DateTimeInterface $publishUp,
		array             $cats,
		int               $limit = 1,
		bool              $withLockedCategories = false,
	): array
	{
		$articles = [];

		$qb = $this->getEr()->createQueryBuilder('a')
			->innerJoin('a.categories', 'aic');
		$qb = $this->setPublishedCriteria($qb, $withLockedCategories);

		$qb->andWhere('a.publishUp < :publishUp')->setParameter('publishUp', $publishUp)
			->andWhere('aic.category IN (:cats)')->setParameter('cats', $cats)
			->orderBy('a.publishUp', 'DESC')
			->select('a.id')
			->setMaxResults($limit);

		foreach ($qb->getQuery()->getArrayResult() as $row) {
			$a = $this->get($row['id']);
			if ($a) {
				$articles[$row['id']] = $a;
			}
		}

		return $articles;
	}

	/**
	 * @throws \Doctrine\DBAL\Exception
	 */
	public function addShow(int $articleId): void
	{
		$this->em->getConnection()
			->executeQuery("UPDATE blog__article SET shows = shows + 1 WHERE id = ?", [$articleId]);
	}

	public function normalizeHydrateArray(array $article): array
	{
		$result = $article[0];
		foreach (['createdBy', 'gallery'] as $v) {
			$result[$v] = $article[$v] ?? null;
		}
		$result['texts'] = $article[0]['texts'][$this->translator->getLocale()];
		foreach (['gallery', 'catId'] as $v) {
			$result[$v] = $article[$v];
		}

		return $result;
	}

	public function getCategoryRootForSite(string $siteIdent): ?int
	{
		if ($this->cRoot === null) {
			$this->cRoot = $this->getCache()->load('categoryRootForSite', function(&$dep) {
				$dep = $this->cacheDep;
				$ids = [];

				foreach ($this->em->createQueryBuilder()->select('IDENTITY(ct.category) as id, ct.alias as alias')
					         ->from(CategoryText::class, 'ct')
					         ->innerJoin('ct.category', 'c', Join::WITH, 'c.lvl = 0')
					         ->getQuery()->getArrayResult() as $row) {
					$ids[$row['alias']] = (int) $row['id'];
				}

				return $ids;
			});
		}

		return $this->cRoot[$siteIdent] ?? null;
	}

	public function findNavigationId(int $categoryId): ?int
	{
		if (!$this->cNavs) {
			$this->cNavs = $this->em->getRepository(Navigation::class)->createQueryBuilder('n')
				->select('n.id, n.componentParams')
				->where('n.componentType IN (:type)')->setParameter('type', ['blog.navigation.category',
					'blog.navigation.blog'])
				->getQuery()->getArrayResult();
		}

		foreach ($this->cNavs as $row) {
			if ($row['componentParams']['category'] && $row['componentParams']['category'] == $categoryId) {
				return (int) $row['id'];
			}

			if ($row['componentParams']['categories'] && in_array($categoryId, $row['componentParams']['categories'])) {
				return (int) $row['id'];
			}
		}

		return null;
	}

	public function fillDao(array $data): Dao\Article
	{
		$text = $data['texts'];

		$dao              = new Dao\Article;
		$dao->id          = (int) $data['id'];
		$dao->lang        = $this->translator->getLocale();
		$dao->title       = $text['title'];
		$dao->alias       = $text['alias'];
		$dao->introtext   = $text['introtext'];
		$dao->fulltext    = $text['fulltext'];
		$dao->categoryId  = (int) $data['catId'];
		$dao->created     = $text['created'];
		$dao->modified    = $text['modified'];
		$dao->featured    = (int) $data['featured'];
		$dao->params      = $data['params'] ?: [];
		$dao->galleryId   = $data['gallery'] ? (int) $data['gallery'] : null;
		$dao->publishUp   = $data['publishUp'];
		$dao->publishDown = $data['publishDown'];
		$dao->createdById = $data['createdBy'] ? (int) $data['createdBy'] : null;
		$dao->layout      = $data['layout'] ?: null;
		$dao->seo         = $text['seo'];

		if (BlogConfig::load('article.allowVideoField')) {
			$dao->video = $data['video'] ?: null;
		}

		if (BlogConfig::load('allowTags')) {
			if ($this->cTags === null) {
				$this->cTags = $this->tagsService->getByAll();
			}

			foreach (array_map('trim', explode(',', $data['tagIds'] ?? '')) as $k) {
				$key = (int) $k;
				if (array_key_exists($key, $this->cTags)) {
					$dao->tags[] = $this->cTags[$key];
				}
			}
		}

		return $dao;
	}

}
