<?php declare(strict_types = 1);

namespace Blog\FrontModule\Model;

use Blog\Model\Entities\ArticleInCategory;
use Blog\Model\Entities\Category;
use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Sites;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\Query\Parameter;
use Nette\Caching\Cache;
use Throwable;

class Categories extends BaseFrontEntityService
{
	/** @var string */
	protected       $entityClass = Category::class;
	protected array $cCategories = [];
	final public const CACHE_NAMESPACE = 'categories';

	protected array $cacheDep = [
		Cache::Tags    => ['categories'],
		Cache::Expire  => '1 month',
		Cache::Sliding => true,
	];

	public function __construct(protected Sites $sites)
	{
	}

	protected function getCache(): Cache
	{
		if ($this->cache === null) {
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);
		}

		return $this->cache;
	}

	public function get(int $id): ?Dao\Category
	{
		/** @var Dao\Category|null $result */
		$result = $this->getAll()[$id] ?? null;

		return $result;
	}

	/**
	 * @return Dao\Category[]|Dao\Category[][]
	 * @throws Throwable
	 */
	public function getAll(?string $rootIdent = null): array
	{
		if (!$rootIdent) {
			$rootIdent = $this->sites->getCurrentSite()->getIdent();
		}
		$key      = $rootIdent . '/' . $this->translator->getLocale();
		$cacheKey = 'categories/' . $this->translator->getLocale();

		if (!$this->cCategories[$key]) {
			$list = $this->getCache()->load($cacheKey, function(&$dep) {
				$dep        = $this->cacheDep;
				$tmp        = [];
				$roots      = [];
				$counts     = [];
				$categories = [];
				$lang       = $this->translator->getLocale();

				foreach ($this->em->getRepository(ArticleInCategory::class)->createQueryBuilder('aic')
					         ->select('COUNT(aic.article) as articles, IDENTITY(aic.category) as category')
					         ->groupBy('aic.category')->getQuery()->getArrayResult() as $row) {
					/** @var array $row */
					$counts[$row['category']] = (int) $row['articles'];
				}

				foreach ($this->getEr()->createQueryBuilder('c')
					         ->select(
						         'c.id, IDENTITY(c.parent) as parent, IDENTITY(c.root) as root, c.lvl, ct.title, ct.alias, ct.text',
					         )
					         ->innerJoin('c.texts', 'ct', Join::WITH, 'ct.lang = :lang')
					         ->orderBy('c.root, c.lft')
					         ->setParameters(new ArrayCollection([new Parameter('lang', $lang)]))
					         ->getQuery()->getArrayResult() as $row) {
					$dao                = new Dao\Category(
						$row['id'],
						$lang,
						$row['title'],
						$row['alias'],
					);
					$dao->id            = $row['id'];
					$dao->lang          = $lang;
					$dao->title         = $row['title'];
					$dao->alias         = $row['alias'];
					$dao->articlesCount = $counts[$row['id']] ?? 0;
					$dao->lvl           = (int) $row['lvl'];
					$dao->rootId        = (int) $row['root'];
					$dao->parentId      = (int) $row['parent'];
					$dao->text          = (string) $row['text'];

					$tmp[$row['id']] = $dao;
					if ($dao->lvl === 0) {
						$roots[$dao->rootId] = $dao->alias;
					}
				}

				foreach ($tmp as $dao) {
					if (isset($roots[$dao->rootId])) {
						$categories[$roots[$dao->rootId]][$dao->getId()] = $dao;
					}
				}

				return $categories;
			});

			$this->cCategories[$key] = $rootIdent === 'all' ? $list : $list[$rootIdent];
		}

		return $this->cCategories[$key] ?: [];
	}

	/**
	 * @param int[] $ids
	 *
	 * @return Dao\Category[]
	 * @throws Throwable
	 */
	public function getCategories(array $ids): array
	{
		$result = [];

		foreach ($this->getAll() as $row) {
			/** @var Dao\Category $row */
			if (in_array($row->getId(), $ids)) {
				$result[$row->getId()] = $row;
			}
		}

		return $result;
	}

	public function getRootCategories(): array
	{
		$result = [];

		foreach ($this->getAll('all') as $rows) {
			/** @var Dao\Category[] $rows */
			foreach ($rows as $row) {
				if ($row->lvl === 0) {
					$result[$row->alias] = $row;
				}
			}
		}

		return $result;
	}

	public function getRootCategory(): ?Dao\Category
	{
		return $this->getRootCategories()[$this->sites->getCurrentSite()->getIdent()];
	}

	public function getFlatTree(int $maxLvl = 1): array
	{
		/** @var Dao\Category[] $cats */
		$cats = $this->getAll();
		$flat = [];

		foreach ($cats as $cat) {
			if ($cat->lvl > $maxLvl) {
				continue;
			}

			$flat[] = [
				'id'     => $cat->getId(),
				'name'   => $cat->title ?: $cat->alias,
				'parent' => $cat->parentId,
			];
		}

		return $flat;
	}

	public function getOptionsForSelect(): array
	{
		/** @var Dao\Category[] $cats */
		$cats   = $this->getAll();
		$result = [];

		foreach ($cats as $cat) {
			if ($cat->parentId !== 0) {
				$result[$cat->getId()] = $cat->title;
			}
		}

		return $result;
	}

}
