<?php declare(strict_types = 1);

namespace Blog\Model\Entities;

use Core\Model\Entities\TId;
use Core\Model\Entities\TranslateListener;
use Core\Model\Entities\TTranslateListener;
use Core\Model\Helpers\Traits\TExtraField;
use DateTimeInterface;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Gallery\Model\Entities\Album;
use Gallery\Model\Entities\Image;
use Tags\Model\Entities\Tag;

#[ORM\Table(name: 'blog__article')]
#[ORM\Entity]
#[ORM\EntityListeners([ArticleListener::class, TranslateListener::class])]
class Article
{
	use TId;
	use TTranslateListener;
	use TExtraField;

	public const EXTRA_FIELD_SECTION = 'blogArticle';

	/**
	 * @var Collection<int, ArticleInCategory>
	 */
	#[ORM\OneToMany(targetEntity: ArticleInCategory::class, mappedBy: 'article', indexBy: 'category')]
	public Collection $categories;

	/**
	 * @var Collection<int, Tag>
	 */
	#[ORM\JoinTable(name: 'blog__article_tags')]
	#[ORM\JoinColumn(name: 'article_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\InverseJoinColumn(name: 'tag_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\ManyToMany(targetEntity: Tag::class, fetch: 'EXTRA_LAZY')]
	public Collection $tags;

	#[ORM\Column(type: Types::STRING, length: 255, nullable: true)]
	protected ?string $image = null;

	#[ORM\Column(type: Types::STRING, length: 255, nullable: true)]
	public ?string $video = null;

	#[ORM\OneToOne(targetEntity: Album::class)]
	#[ORM\JoinColumn(name: 'gallery_id', referencedColumnName: 'id', onDelete: 'SET NULL')]
	public ?Album $gallery = null;

	#[ORM\Column(type: Types::DATETIME_MUTABLE, nullable: true)]
	public ?DateTimeInterface $publishUp = null;

	#[ORM\Column(type: Types::DATETIME_MUTABLE, nullable: true)]
	public ?DateTimeInterface $publishDown = null;

	#[ORM\Column(type: Types::STRING, length: 60, nullable: true)]
	private ?string $featured = null;

	#[ORM\Column(type: Types::JSON, nullable: true)]
	public ?array $params = null;

	#[ORM\Column(type: Types::INTEGER, options: ['unsigned' => true, 'default' => 0])]
	public int $shows;

	/**
	 * @var Collection<string, ArticleText>
	 */
	#[ORM\OneToMany(targetEntity: ArticleText::class, mappedBy: 'article', indexBy: 'lang')]
	protected Collection $texts;

	#[ORM\Column(type: Types::STRING, length: 60, nullable: true)]
	public ?string $layout = null;

	public function __construct()
	{
		$this->texts       = new ArrayCollection;
		$this->shows       = 0;
		$this->categories  = new ArrayCollection;
		$this->extraFields = new ArrayCollection;
		$this->tags        = new ArrayCollection;
	}

	public function setTexts(array $texts): self
	{
		$this->texts = new ArrayCollection($texts);

		return $this;
	}

	public function getText(?string $lang = null): ?ArticleText { return $this->texts[$lang ?: $this->locale] ?? null; }

	/**
	 * @return Collection<string, ArticleText>
	 */
	public function getTexts(): Collection
	{
		return $this->texts;
	}

	public function setId(?int $id): void
	{
		$this->id = $id;
	}

	/******
	 * === Image
	 */

	public function getImage(): ?Image
	{
		return $this->gallery && $this->gallery->getCoverImage() ? $this->gallery->getCoverImage() : null;
	}

	public function getImageFile(): ?string
	{
		return $this->getImage() ? $this->getImage()->getFilePath() : null;
	}

	public function getGallery(): ?Album
	{
		return $this->gallery;
	}

	/*******
	 * === Featured
	 */

	public function getFeatured(): ?string
	{
		return $this->featured;
	}

	public function getIsFeatured(): bool
	{
		return $this->featured !== null;
	}

	public function setFeatured(?string $featured): self
	{
		$this->featured = $featured;

		return $this;
	}

	/******
	 * === Tags
	 */

	public function setTags(array $tags): void
	{
		$this->tags = new ArrayCollection($tags);
	}

	/**
	 * @return Collection<Tag>
	 */
	public function getTags(): Collection
	{
		return $this->tags;
	}

	public function getTagsId(): array
	{
		return $this->tags->map(fn(Tag $tag) => $tag->getId())->toArray();
	}

	public function getParams(): array { return $this->params ?: []; }

	/**
	 * @return mixed|null $key
	 */
	public function getParam(string $key) { return $this->params[$key] ?? null; }

	/**
	 * @param mixed|null $value
	 */
	public function setParam(string $key, $value): self
	{
		if (!is_array($this->params)) {
			$this->params = [];
		}

		if ($value) {
			$this->params[$key] = $value;
		} else {
			unset($this->params[$key]);
		}

		return $this;
	}

	/**
	 * @return int[]
	 */
	public function getTagIds(): array
	{
		return $this->tags->map(fn(Tag $tag) => $tag->getId())->toArray();
	}

}
