<?php declare(strict_types = 1);

namespace Blog\Model\Entities;

use Core\Model\Entities\TSeo;
use DateTimeInterface;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Nette\Utils\Strings;
use Users\Model\Entities\User;

#[ORM\Table(name: 'blog__article_text')]
#[ORM\Index(name: 'article', columns: ['article_id', 'lang'])]
#[ORM\Entity]
class ArticleText
{
	use TSeo;

	#[ORM\Id]
	#[ORM\ManyToOne(targetEntity: Article::class, inversedBy: 'texts')]
	#[ORM\JoinColumn(name: 'article_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	public Article $article;

	#[ORM\Id]
	#[ORM\Column(type: Types::STRING, length: 2, nullable: true)]
	protected ?string $lang = null;

	#[ORM\Column(type: Types::STRING, length: 255)]
	protected string $title;

	#[ORM\Column(type: Types::STRING, length: 255)]
	protected ?string $alias = null;

	#[ORM\Column(type: Types::SMALLINT)]
	public int $isPublished;

	#[ORM\Column(type: Types::TEXT, nullable: true)]
	public ?string $introtext = null;

	#[ORM\Column(name: '`text`', type: Types::TEXT, nullable: true)]
	public ?string $fulltext = null;

	#[Gedmo\Timestampable(on: 'create')]
	#[ORM\Column(name: 'created', type: Types::DATETIME_MUTABLE)]
	protected ?DateTimeInterface $created = null;

	#[ORM\ManyToOne(targetEntity: User::class)]
	#[ORM\JoinColumn(name: 'created_by', referencedColumnName: 'id', onDelete: 'SET NULL')]
	public ?User $createdBy = null;

	#[Gedmo\Timestampable(on: 'update')]
	#[ORM\Column(name: 'modified', type: Types::DATETIME_MUTABLE, nullable: true)]
	protected ?DateTimeInterface $modified = null;

	#[ORM\ManyToOne(targetEntity: User::class)]
	#[ORM\JoinColumn(name: 'modified_by', referencedColumnName: 'id', nullable: true, onDelete: 'SET NULL')]
	public ?User $modifiedBy = null;

	public function __construct(Article $article, string $title, string $lang)
	{
		$this->article = $article;
		$this->lang    = $lang;
		$this->setTitle($title);
		$this->setSeo([]);
	}

	public function getTitle(): ?string { return $this->title; }

	public function setTitle(string $title): self
	{
		$this->title = $title;
		if (!$this->alias) {
			$this->setAlias($title);
		}

		return $this;
	}

	public function getAlias(): ?string { return $this->alias ?: Strings::webalize($this->title); }

	public function setAlias(string $alias): self
	{
		$this->alias = Strings::webalize($alias);

		return $this;
	}

	public function getCreated(): DateTimeInterface { return $this->created; }

	public function getModified(): DateTimeInterface { return $this->modified; }
}
