<?php declare(strict_types = 1);

namespace Blog\Model\Entities;

use Core\Model\Entities\Repository\NestedTreeRepository;
use Core\Model\Entities\TId;
use Core\Model\Entities\TranslateListener;
use Core\Model\Entities\TTranslateListener;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;

#[ORM\Table(name: 'blog__category')]
#[Gedmo\Tree(type: 'nested')]
#[ORM\Entity(repositoryClass: NestedTreeRepository::class)]
#[ORM\EntityListeners([CategoryListener::class, TranslateListener::class])]
class Category
{
	use TId;
	use TTranslateListener;

	#[ORM\Column(type: Types::STRING, length: 60)]
	public string $layout;

	#[ORM\Column(type: Types::STRING, length: 60, nullable: true)]
	public ?string $password = null;

	#[ORM\Column(type: Types::SMALLINT, options: ['default' => 1])]
	public int $isPublished = 0;

	/**
	 * @var Collection<int, Category>
	 */
	#[ORM\OneToMany(targetEntity: Category::class, mappedBy: 'parent')]
	#[ORM\OrderBy(['lft' => 'ASC'])]
	public Collection $children;

	#[Gedmo\TreeParent]
	#[ORM\ManyToOne(targetEntity: Category::class, cascade: ['persist'], inversedBy: 'childrens')]
	#[ORM\JoinColumn(name: 'parent_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	public ?Category $parent = null;

	#[Gedmo\TreeLeft]
	#[ORM\Column(name: 'lft', type: 'integer')]
	protected int $lft;

	#[Gedmo\TreeLevel]
	#[ORM\Column(name: 'lvl', type: 'integer')]
	protected int $lvl;

	#[Gedmo\TreeRight]
	#[ORM\Column(name: 'rgt', type: 'integer')]
	protected int $gt;

	#[Gedmo\TreeRoot]
	#[ORM\ManyToOne(targetEntity: Category::class)]
	#[ORM\JoinColumn(name: 'tree_root', referencedColumnName: 'id', onDelete: 'CASCADE')]
	public ?Category $root;

	/**
	 * @var Collection<string, CategoryText>
	 */
	#[ORM\OneToMany(targetEntity: CategoryText::class, mappedBy: 'category', indexBy: 'lang')]
	protected Collection $texts;

	public function __construct()
	{
		$this->texts       = new ArrayCollection;
		$this->layout      = 'default';
		$this->children    = new ArrayCollection;
		$this->isPublished = 1;
	}

	public function getLvl(): int { return $this->lvl; }

	public function getRoot(): ?Category { return $this->root; }

	public function setPassword(?string $password): void
	{
		$this->password = $password ?: null;
	}

	public function isLocked(): bool { return $this->password ? true : false; }

	public function getLayout(): string { return $this->layout ?: 'default'; }

	public function setLayout(string $layout): self
	{
		$this->layout = $layout;

		return $this;
	}

	public function setTexts(array $texts): self
	{
		$this->texts = new ArrayCollection($texts);

		return $this;
	}

	public function getText(?string $lang = null): ?CategoryText
	{
		return $this->texts[$lang ?: $this->locale] ?? null;
	}

	/**
	 * @return Collection<string, CategoryText>
	 */
	public function getTexts(): Collection { return $this->texts; }
}
