<?php declare(strict_types = 1);

namespace Blog\Model\Subscribers;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Parameter;
use Blog\FrontModule\Model\Articles;
use Blog\Model\BlogConfig;
use Blog\Model\Entities\Article;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Entities\ExtraField;
use Navigations\Model\Event\RouteInEvent;
use Navigations\Model\Navigations;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class NavigationSubscriber implements EventSubscriberInterface
{
	public function __construct(
		protected EntityManagerDecorator $entityManager,
		protected Articles               $articles,
		protected Navigations            $navigations,
	)
	{
	}

	public static function getSubscribedEvents(): array
	{
		return [
			Navigations::class . '::routeInNotFound' => ['routeInNotFound', 110],
		];
	}

	public function routeInNotFound(RouteInEvent $event): void
	{
		if (!BlogConfig::load('allowArticleFullUrlField', false)) {
			return;
		}

		$articleId = $this->entityManager->getRepository(ExtraField::class)->createQueryBuilder('ef')
			->select('ef.sectionKey')
			->where('ef.sectionName = :sn')
			->andWhere('ef.key = :key')
			->andWhere('ef.value = :value')
			->andWhere('ef.lang = :lang')
			->setParameters(new ArrayCollection([new Parameter('sn', Article::EXTRA_FIELD_SECTION), new Parameter('key', 'fullUrl'), new Parameter('value', $event->urlParams['path']), new Parameter('lang', $event->urlParams['locale'])]))->getQuery()->getScalarResult()[0]['sectionKey'] ?? null;

		if ($articleId) {
			$articleId = (int) $articleId;
			$article   = $this->articles->get($articleId);

			if ($article) {
				$nav = $this->articles->findNavigationId($article->categoryId);

				if ($nav) {
					$nav                               = $this->navigations->getNavigation($nav);
					$event->return['presenter']        = 'Blog:Front:Articles';
					$event->return['action']           = 'detail';
					$event->return['id']               = $articleId;
					$event->return['activeNavigation'] = $nav;
					$event->stopPropagation();
				}
			}
		}
	}
}
