<?php declare(strict_types = 1);

namespace Blog\AdminModule\Components\Article;

use Blog\AdminModule\Model\Categories;
use Blog\Model\Articles;
use Blog\Model\BlogCache;
use Blog\Model\BlogConfig;
use Blog\Model\Entities\Article;
use Blog\Model\Entities\ArticleInCategory;
use Blog\Model\Entities\ArticleText;
use Blog\Model\Entities\Category;
use Blog\Model\OpenedArticlesService;
use Core\AdminModule\Model\Sites;
use Core\Components\FilesManager\FilesManager;
use Core\Components\FilesManager\IFilesManagerFactory;
use Core\FrontModule\Model\SeoContainer;
use Core\Model\Entities\ExtraField;
use Core\Model\Event\ComponentTemplateEvent;
use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\FormValidateEvent;
use Core\Model\Event\SetFormDataEvent;
use Core\Model\Helpers\EntityHelpers;
use Core\Model\Helpers\Strings;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Core\Model\UI\Form\BootstrapRenderer;
use Exception;
use Gallery\AdminModule\Components\Image\IImagesZoneFactory;
use Gallery\AdminModule\Components\Image\ImagesZone;
use Gallery\Model\Entities\Album;
use Nette\Application\IPresenter;
use Nette\Http\Request;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;
use Nette\Utils\DateTime;
use Tags\Model\Tags;
use Users\Model\Entities\Acl;
use Users\Model\Entities\User;

class ArticleForm extends BaseControl
{
	public ?Article                 $article = null;
	private Categories              $categoriesService;
	private Articles                $articlesService;
	private IImagesZoneFactory      $imagesZoneFactory;
	private IFilesManagerFactory    $filesManagerFactory;
	protected OpenedArticlesService $openedArticlesService;
	protected Request               $httpRequest;
	protected SeoContainer          $seoContainerService;
	protected Sites                 $sitesService;
	protected BlogCache             $blogCache;
	protected Tags                  $tagsService;

	public function __construct(
		Articles              $articles,
		Categories            $categories,
		Sites                 $sites,
		IFilesManagerFactory  $filesManagerFactory,
		IImagesZoneFactory    $imagesZoneFactory,
		OpenedArticlesService $openedArticlesService,
		Request               $httpRequest,
		SeoContainer          $seoContainer,
		BlogCache             $blogCache,
		Tags                  $tagsService
	)
	{
		$this->articlesService       = $articles;
		$this->categoriesService     = $categories;
		$this->sitesService          = $sites;
		$this->filesManagerFactory   = $filesManagerFactory;
		$this->imagesZoneFactory     = $imagesZoneFactory;
		$this->openedArticlesService = $openedArticlesService;
		$this->httpRequest           = $httpRequest;
		$this->seoContainerService   = $seoContainer;
		$this->blogCache             = $blogCache;
		$this->tagsService           = $tagsService;

		$this->monitor(IPresenter::class, function() {
			if ($this->article) {
				$this->template->article = $this->article;
			}
			if (!$this['form']->getComponent('author')->value) {
				$this['form']->getComponent('author')->setDefaultValue($this->presenter->getUser()->getId());
			}
		});
	}

	public function render(): void
	{
		$this->eventDispatcher->dispatch(new ComponentTemplateEvent($this->template, $this), self::class . '::render');

		$this->template->article  = $this->article;
		$this->template->sites    = $this->sitesService->getAll();
		$this->template->thisForm = $this['form'];
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$roles = [];
		foreach ($this->em->getRepository(Acl::class)->createQueryBuilder('acl')
			         ->select('IDENTITY(acl.role) as role')
			         ->join('acl.privilege', 'pr', 'WITH', 'pr.name IN (:privilegeName)')
			         ->join('acl.resource', 'res', 'WITH', 'res.name IN (:resourceName)')
			         ->andWhere('acl.allowed = 1')
			         ->setParameters([
				         'privilegeName' => ['access', 'all'],
				         'resourceName'  => ['Blog:Admin', 'all'],
			         ])->getQuery()->getScalarResult() as $row) {
			$roles[] = $row['role'];
		}

		$users = [];
		foreach ($this->em->getRepository(User::class)->createQueryBuilder('u')
			         ->select('u.id, u.name, u.lastname')
			         ->join('u.roles', 'r', 'WITH', 'r.id IN (:roles)')->setParameter('roles', $roles)
			         ->orderBy('u.name', 'ASC')->getQuery()->getScalarResult() as $row) {
			$users[$row['id']] = trim($row['lastname'] . ' ' . $row['name']);
		}

		$form->addGroup('default.form.general');
		$form->addText('title', 'blog.articleForm.title')->setMaxLength(255)->setIsMultilanguage();
		$form->addText('alias', 'blog.articleForm.alias')
			->setMaxLength(255)
			->setDescription('default.willBeFilledAutomaticaly')
			->setIsMultilanguage();
		$form->addSelect('author', $this->t('blog.articleForm.author'), $users)->setTranslator(null)->setIsMultilanguage();
		$form->addBool('isPublished', 'blog.articleForm.isPublished')->setDefaultValue(1)->setIsMultilanguage();

		if (BlogConfig::load('allowTags')) {
			$form->addCheckboxList('tags', 'blog.articleForm.tags', $this->tagsService->getOptionsForSelect());
		}

		if (BlogConfig::load('article.allowVideoField')) {
			$form->addText('video', 'blog.article.videoUrl')
				->setMaxLength(255);
		}

		$layouts = [null => 'default'];
		foreach (glob(TEMPLATES_DIR . '/Front/default/Blog/Articles/Components/ArticleDetail/*.latte') ?: [] as $file) {
			$fileName           = pathinfo($file, PATHINFO_FILENAME);
			$layouts[$fileName] = $fileName;
		}
		$form->addSelect('layout', 'blog.article.layout', $layouts);

		$form->addTextArea('introtext', 'blog.articleForm.introtext')
			->setMaxLength((int) BlogConfig::load('article.introtextLength'))
			->setDescription('blog.articleForm.introtextWillBeFilledAutomaticaly')
			->setIsMultilanguage();
		$form->addEditor('fulltext', 'blog.articleForm.fulltext')->setToolbar('Blog')
			->setAlbumImagesCallback([$this, 'albumImagesCallback'])->setDisableAutoP(false)->setIsMultilanguage();
		$form->addDateTimePicker('publishUp', 'blog.articleForm.publishUp')->setDefaultValue(new DateTime)->setRequired();
		$form->addDateTimePicker('publishDown', 'blog.articleForm.publishDown');
		$form->addText('source', 'blog.articleForm.source');
		$form->addTextArea('footer', 'blog.articleForm.footer');

		$categoriesContainer = new BaseContainer;
		$sites               = $this->sitesService->getAll();
		foreach ($sites as $site) {
			$categoryContainer = new BaseContainer;

			$trees = $this->categoriesService->getTreeForSelect($site->getIdent());

			$categoryContainer->addCustomData('trees', $trees);
			$categoryContainer->addCustomData('template', __DIR__ . '/ArticleForm_categoryContainer.latte');

			$categoryContainer->addRadioList('category', 'eshopCatalog.productForm.defaultCategory',
				$this->categoriesService->getOptionsForSelect($site->getIdent()));

			$categoriesContainer->addComponent($categoryContainer, $site->getIdent());
		}
		$form->addComponent($categoriesContainer, 'categories');

		if (!$this->article && isset($trees) && isset($site) && count($sites) === 1 && count($trees) === 1 && count(array_values($trees)[0]['children']) === 1) {
			$form->getComponent('categories')[$site->getIdent()]['category']->setDefaultValue(array_values($trees)[0]['children'][0]['id']);
		}

		$form->addHidden('articleId');
		$form->addHidden('preparedAlbumId');

		$form->addComponent($this->seoContainerService->getContainer(true), 'seo');

		// Rozšířené
		if (BlogConfig::load('allowArticleFullUrlField', false)) {
			$form->addText('fullUrl', 'blog.articleForm.fullUrl')
				->setDescription('blog.articleForm.fullUrlDesc')
				->setIsMultilanguage();

			/** @var BootstrapRenderer $renderer */
			$renderer = $form->getRenderer();
			$renderer->addToBaseExtendedLayout('right', 'fullUrl');
		}

		$this->eventDispatcher->dispatch(new CreateFormEvent($form, null), ArticleForm::class . '::createForm');

		$form->addSaveCancelControl();
		$form->addSubmit('submitAndClose', 'default.saveAndClose');
		$form->addSubmit('submit', 'default.save');

		$form->onValidate[] = [$this, 'formValidate'];
		$form->onSuccess[]  = [$this, 'formSuccess'];

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values): void
	{
		$presenter = $this->presenter;
		$this->eventDispatcher->dispatch(
			new FormValidateEvent($form, $values, $this->template, $presenter), self::class . '::validateForm'
		);

		if ($form->hasErrors()) {
			$this->redrawControl('form');
		}
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		$presenter = $this->presenter;
		$user      = $presenter->getUser();
		try {
			$langValues = $form->convertMultilangValuesToArray();
			/** @var ArticleText[] $texts */
			$texts = [];


			if ($this->article) {
				$article = $this->article;
				$texts   = $article->getTexts()->toArray();
			} else if ($values->articleId) {
				$article = $this->articlesService->get((int) $values->articleId);
				$texts   = $article->getTexts()->toArray();
			} else {
				$article = new Article;
			}

			foreach ($langValues as $l => $v) {
				if (!isset($texts[$l]) && $v['title']) {
					$texts[$l] = new ArticleText($article, $v['title'], $l);
				}

				if ($v['title'] == '' || $texts[$l]->getTitle() == '') {
					/** @var ArticleText|null $articleText */
					$articleText = $texts[$l];
					if ($articleText) {
						$this->em->remove($texts[$l]);
					}
					unset($texts[$l]);
					continue;
				}

				$texts[$l]->setTitle($v['title'])
					->setAlias($v['alias'] ?: $v['title'])
					->setSeo($v['seo'] ?? []);

				/** @var User $modifiedBy */
				$modifiedBy = $user->getIdentity();

				/** @var User $createdBy */
				$createdBy = $this->em->getReference(User::class, $v['author']);

				$texts[$l]->isPublished = (int) $v['isPublished'];
				$texts[$l]->introtext   = $v['introtext'];
				$texts[$l]->fulltext    = $v['fulltext'];
				$texts[$l]->createdBy   = $createdBy;
				$texts[$l]->modifiedBy  = $modifiedBy;

				if (!$texts[$l]->introtext) {
					$texts[$l]->introtext = Strings::truncate(strip_tags($texts[$l]->fulltext), (int) BlogConfig::load('article.introtextLength'));
				}

				$this->em->persist($texts[$l]);
			}

			$article->setTexts($texts);
			$article->publishUp   = $values->publishUp;
			$article->publishDown = $values->publishDown;
			$article->layout      = $values->layout;
			$article->params      = [
				'source' => $values->source,
				'footer' => $values->footer,
			];

			if (BlogConfig::load('article.allowVideoField')) {
				$article->video = $values->video ?: null;
			}

			if (BlogConfig::load('allowTags')) {
				$addTags = array_diff($values->tags, $article->getTagIds());
				$removeTags = array_diff($article->getTagIds(), $values->tags);

				foreach ($addTags as $tagId) {
					if (($tag = $this->tagsService->get((int) $tagId)) && !$article->tags->contains($tag)) {
						$article->tags->add($tag);
					}
				}
				foreach ($removeTags as $tagId) {
					if (($tag = $this->tagsService->get((int) $tagId)) && $article->tags->contains($tag)) {
						$article->tags->removeElement($tag);
					}
				}
			}

			// Extra
			if (BlogConfig::load('allowArticleFullUrlField', false)) {
				foreach ($values->fullUrl as $l => $v) {
					if ($v) {
						$efFullUrl = EntityHelpers::setExtraField($article, 'fullUrl', $v, $l);
						$this->em->persist($efFullUrl);
					} else {
						/** @var ExtraField[] $efs */
						$efs       = $article->getExtraFieldsByKey()['fullUrl'];
						$efFullUrl = $efs[$l] ?? null;

						if ($efFullUrl) {
							$this->em->remove($efFullUrl);
						}
					}
				}
			}

			$this->em->persist($article);


			// Kategorie
			/** @var ArticleInCategory[] $currentCategories */
			$currentCategories = [];
			$formCategoriesIds = [];

			foreach ($article->categories->toArray() as $cia) {
				$currentCategories[$cia->getCategory()->getId()] = $cia;
			}

			foreach ((array) $values->categories as $v) {
				if ($v->category) {
					$formCategoriesIds[$v->category] = $v->category;
				}
			}

			foreach (array_diff_key($formCategoriesIds, $currentCategories) as $catId) {
				/** @var Category $category */
				$category = $this->categoriesService->getReference($catId);

				$articleInCategory = new ArticleInCategory($article, $category);
				$this->em->persist($articleInCategory);
			}

			foreach (array_diff_key($currentCategories, $formCategoriesIds) as $cat) {
				$this->em->remove($cat);
			}

			$this->em->flush();

			$event                   = new FormSuccessEvent($form, $values, $this->template, $this->presenter);
			$event->custom['entity'] = $article;
			$this->eventDispatcher->dispatch($event, self::class . '::formSuccess');

			$form->addCustomData('articleId', $article->getId());
			$this->presenter->flashMessageSuccess('blog.articleForm.articleSaved');
			$this->article = $article;

			$this->blogCache->clearCache($article->getId());
		} catch (Exception $e) {
			if ($this->em->getConnection()->isTransactionActive()) {
				$this->em->rollback();
			}

			$form->addError($e->getMessage());
			$this->presenter->redrawControl('flashes');
			$this->redrawControl('formErrors');
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	/**
	 * @param int|string $id
	 */
	public function setArticle($id): void
	{
		$this->article = $this->em->getRepository(Article::class)->find($id);

		if (!$this->article) {
			throw new InvalidArgumentException;
		}

		$form = $this['form'];
		$a    = $this->article;
		$d    = [];
		$form->setDefaults([
			'publishUp'       => $a->publishUp,
			'publishDown'     => $a->publishDown,
			'articleId'       => $a->getId(),
			'preparedAlbumId' => $a->getGallery() ? $a->getGallery()->getId() : '',
			'layout'          => $a->layout,
			'video'           => $a->video,
		]);

		if (BlogConfig::load('allowTags')) {
			$form->setDefaults(['tags' => $a->getTagIds()]);
		}

		foreach ($a->getTexts() as $l => $text) {
			$d['title'][$l]       = $text->getTitle();
			$d['alias'][$l]       = $text->getAlias();
			$d['isPublished'][$l] = $text->isPublished;
			$d['introtext'][$l]   = $text->introtext;
			$d['fulltext'][$l]    = $text->fulltext;

			if ($text->createdBy && array_key_exists($text->createdBy->getId(), $form->getComponent('author')->getItems())) {
				$d['author'][$l] = $text->createdBy->getId();
			}
		}

		if (is_array($a->params)) {
			foreach ($a->params as $k => $v) {
				if (isset($form[$k]) && $form[$k] instanceof \Nette\Forms\Controls\BaseControl) {
					$type = $form[$k]->getOption('type');

					if ($type == 'select') {
						if (!array_key_exists($v, $form->getComponent($k)->getItems())) {
							continue;
						}
					}
					$form[$k]->setDefaultValue($v);
				}
			}
		}

		// Kategorie
		$allCategories = array_map(static fn(ArticleInCategory $v) => $v->getCategory()->getId(), $a->categories->toArray());

		foreach ($form->getComponent('categories')->getComponents() as $ident => $c) {
			$items = array_keys($c['category']->getItems());

			foreach ($allCategories as $cat) {
				if (in_array($cat, $items)) {
					$form->getComponent('categories')[$ident]['category']->setValue($cat);

					break;
				}
			}
		}

		// Extra
		$ef    = $a->getExtraFieldsValues();
		$extra = [];

		if (BlogConfig::load('allowArticleFullUrlField', false)) {
			$extra['fullUrl'] = $ef['fullUrl'] ?? '';
		}
		$form->setDefaults($extra);

		$this->seoContainerService->setDefaultsFromEntity($form->getComponent('seo'), $a->getTexts()->toArray());
		$form->setDefaults($d);

		$this->eventDispatcher->dispatch(new SetFormDataEvent($form, $this->article), ArticleForm::class . '::setArticle');
	}

	public function albumImagesCallback(): string
	{
		return 'preparedAlbumId';
	}

	/*******************************************************************************************************************
	 * =================  Handle
	 */

	// TODO odstranit
	public function handlePreSave(): void
	{
		$presenter = $this->presenter;
		$data      = $this->httpRequest->getPost('formData');
		if ($this->article || !$data) {
			$presenter->sendPayload();
		}

		try {
			$article            = new Article();
			$articleText        = new ArticleText($article, $data['title'], $this->translator->getLocale());
			$article->publishUp = new DateTime();
			$this->em->persist($article);
			$this->em->persist($articleText);
			$this->em->flush();
			$presenter->payload->editUrl = $presenter->link('edit', [$article->getId()]);
			$presenter->payload->article = [
				'id'    => $article->getId(),
				'alias' => $articleText->getAlias(),
			];
			$presenter->flashMessageSuccess('blog.article.articleCreated');
		} catch (Exception $e) {
			$presenter->payload->error = true;
			$presenter->flashMessageDanger('blog.article.articleCreatingError');
		}

		$presenter->redrawControl('flashes');
	}

	public function handleSaveProgress(): void
	{
		$dataRaw = $this->httpRequest->getPost('formData');
		$data    = [];

		if (!$dataRaw) {
			$this->presenter->sendPayload();
		}

		foreach ($dataRaw as $v) {
			if (in_array($v, ['_token_', '_do'])) {
				continue;
			}
			$data[$v['name']] = $v['value'];
		}

		if (!$this->article) {
			$this->article = $this->articlesService->getReference((int) $data['articleId']);
		}

		if (!$this->article) {
			$this->presenter->sendPayload();
		}

		$this->presenter->flashMessageSuccess('blog.article.autoSaveSuccess');
		$this->presenter->redrawControl('flashes');
	}

	public function handleTagsAutocomplete(string $q): void
	{
		$tags = [];
		foreach ($this->tagsService->filterAvailableTags($q) as $tag) {
			$tags[] = ['key' => $tag->getId(), 'value' => $tag->title];
		}

		$this->presenter->sendJson($tags);
	}

	public function handleCreateTag(string $value): void
	{
		$this->presenter->sendJson($this->tagsService->createTag($value));
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentFilesManager(): FilesManager
	{
		return $this->filesManagerFactory->create();
	}

	protected function createComponentImagesZone(): ImagesZone
	{
		$control = $this->imagesZoneFactory->create();

		if ($this->article && $this->article->gallery) {
			$control->setAlbum($this->article->gallery->getId());
		}

		$control->onEmpty[] = function($control) {
			/** @var ImagesZone $control */
			$dataRaw = $this->httpRequest->getPost('formData');
			$data    = [];

			if (!$dataRaw) {
				return false;
			}

			foreach ($dataRaw as $row) {
				$data[$row['name']] = $row['value'];
			}

			$this->article = $this->articlesService->get((int) $data['articleId']);

			if (!$this->article) {
				$article         = new Article;
				$article->locale = $this->translator->getLocale();
				$this->em->persist($article);

				$keySuffix                = '[' . $this->translator->getLocale() . ']';
				$articleText              = new ArticleText($article, $data['title' . $keySuffix], $this->translator->getLocale());
				$articleText->isPublished = (int) $data['isPublished' . $keySuffix];
				$article->publishUp       = DateTime::createFromFormat('j. n. Y - H:i', $data['publishUp']) ?: new DateTime;

				$this->em->persist($articleText);
				$article->getTexts()->set($this->translator->getLocale(), $articleText);

				$this->em->persist($article);

				$this->em->flush();
				$this->article = $article;
			}

			if (!$this->article->gallery) {
				$album        = new Album(UPLOADS_PATH . '/gallery');
				$album->title = $data['title'];
				$this->em->persist($album);
				$this->em->flush();

				$control->setAlbum($album->getId());
				$this->article->gallery = $album;

				$this->em->persist($this->article);
				$this->em->flush();
			}

			$control->extendedResponseOnEmpty['articleId']    = $this->article->getId();
			$control->extendedResponseOnEmpty['articleTitle'] = $this->article->getText()->getTitle();
			$control->extendedResponseOnEmpty['articleAlias'] = $this->article->getText()->getAlias();

			return true;
		};

		return $control;
	}
}
