<?php declare(strict_types = 1);

namespace Blog\FrontModule\Components\Article;

use Blog\FrontModule\Model\Articles;
use Blog\Model\ArticlesFacade;
use Blog\Model\Categories;
use Blog\FrontModule\Model\Dao\Article;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Core\Model\UI\FrontPresenter;
use Gallery\FrontModule\Components\GalleryControl;
use Gallery\FrontModule\Components\IGalleryControlFactory;
use Navigations\Model\Navigations;
use Nette\Caching\Cache;
use Nette\Http\Session;
use Nette\Http\SessionSection;
use Users\Model\Users;

class ArticleDetail extends BaseControl
{
	protected ?Article             $article = null;
	protected int                  $page    = 1;
	protected Articles             $articlesService;
	protected Users                $usersService;
	protected Categories           $categoriesService;
	private IGalleryControlFactory $galleryFactory;
	protected Session              $session;
	protected SessionSection       $sessionSection;
	protected ArticlesFacade       $articlesFacade;
	protected Navigations          $navigations;

	public bool $isLocked = false;
	public int  $limit    = 9;
	public int  $start    = 0;

	public function __construct(
		Articles               $articles,
		Users                  $users,
		Categories             $categories,
		IGalleryControlFactory $galleryFactory,
		Session                $session,
		ArticlesFacade         $articlesFacade,
		Navigations            $navigations
	)
	{
		$this->articlesService   = $articles;
		$this->galleryFactory    = $galleryFactory;
		$this->usersService      = $users;
		$this->categoriesService = $categories;
		$this->session           = $session;
		$this->articlesFacade    = $articlesFacade;
		$this->navigations       = $navigations;

		$this->sessionSection = $session->getSection('categories');
		$this->sessionSection->setExpiration('1 week', 'unlocked');

		$this->monitor(FrontPresenter::class, function(FrontPresenter $presenter) {
			$this->cache = new Cache($this->cacheStorage, Articles::CACHE_NAMESPACE);
		});
	}

	public function render(): void
	{
		/** @var FrontPresenter $presenter */
		$presenter = $this->presenter;

		if (!$this->article) {
			return;
		}

		if ($this->article->category->password != null
			&& !($this->sessionSection->unlocked && in_array($this->article->category->getId(), $this->sessionSection->unlocked))
		) {
			$this->isLocked = true;
			$this->template->setFile(__DIR__ . '/password.latte');
			$this->template->render();
		} else {
			$this->isLocked = false;
			$this->articlesService->addShow($this->article->getId());

			$this->template->article      = $this->article;
			$this->template->nextArticles = fn(int $limit = 1) => $this->getNextArticles($limit);
			$this->template->prevArticles = fn(int $limit = 1) => $this->getPrevArticles($limit);
			$this->template->backLink     = $this->navigations->generateLink($presenter->getActiveNavigation());

			if ($this->article->layout && $this->article->layout !== 'default') {
				$templateFile = TEMPLATES_DIR . '/Front/default/Blog/Articles/Components/ArticleDetail/' . $this->article->layout . '.latte';
			} else {
				$templateFile = $this->getTemplateFile();
			}

			$this->template->render($templateFile);
		}
	}

	/**
	 * @param Article|int $article
	 */
	public function setArticle($article): void
	{
		$this->article = $article instanceof Article ? $article : $this->articlesService->get($article);
		if ($this->article->category->password != null
			&& !($this->sessionSection->unlocked && in_array($this->article->category->getId(), $this->sessionSection->unlocked))
		) {
			$this->isLocked = true;
		} else {
			$this->isLocked = false;
		}
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */
	protected function createComponentGallery(): GalleryControl
	{
		$control = $this->galleryFactory->create();
		if ($this->article->gallery) {
			$control->setAlbum($this->article->gallery->id);
		}

		return $control;
	}

	protected function createComponentCategoryPasswordForm(): BaseForm
	{
		$form = $this->createForm();

		$form->addPassword('password', 'blog.categoryPasswordForm.password')
			->setHtmlAttribute('placeholder', $this->t('blog.categoryPasswordForm.password'))
			->setRequired();
		$form->addSubmit('submit', 'blog.categoryPasswordForm.unlock');

		$form->onSubmit[] = function($form) {
			$values = $form->getValues();

			if ($this->article->category->password == $values->password) {
				if (!is_array($this->sessionSection->unlocked)) {
					$this->sessionSection->unlocked = [];
				}
				$this->sessionSection->unlocked[] = $this->article->category->getId();
			}

			$this->redirect('this');
		};

		return $form;
	}

	/**
	 * @return array|array[]
	 */
	public function getNextArticles(int $limit = 1)
	{
		$articles = $this->articlesService->getNextPublishedArticles($this->article->publishUp, [$this->article->categoryId], $limit);

		return $limit === 1 && $articles ? array_values($articles)[0] : $articles;
	}

	/**
	 * @return Article|Article[]
	 */
	public function getPrevArticles(int $limit = 1)
	{
		$articles = $this->articlesService->getPrevPublishedArticles($this->article->publishUp, [$this->article->categoryId], $limit);

		return $limit === 1 && $articles ? array_values($articles)[0] : $articles;
	}

}
