<?php declare(strict_types = 1);

namespace Blog\FrontModule\Components;

use Blog\FrontModule\Model\Articles;
use Blog\FrontModule\Model\Dao\Article;
use Blog\FrontModule\Model\Dao\Category;
use Blog\Model\ArticlesFacade;
use Blog\Model\BlogConfig;
use Blog\FrontModule\Model\Categories;
use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\UI\BaseControl;
use Core\Model\UI\FrontPresenter;
use Gallery\Model\Images;
use Nette\Caching\Cache;
use Nette\Utils\Paginator;
use Throwable;

class ArticlesList extends BaseControl
{
	/** @var Article[]|null */
	protected ?array $articles = null;
	/** @var int[]|null */
	protected ?array            $categories;
	protected string            $sort;
	protected bool              $showLocked;
	protected ?int              $articlesCount    = null;
	protected Articles          $articlesService;
	protected Categories        $categoriesService;
	protected Images            $imagesService;
	protected ArticlesFacade    $articlesFacade;
	protected DaoNavigationItem $activeNavigation;
	public int                  $page             = 1;
	public int                  $limit;
	public int                  $start            = 0;
	public bool                 $paginatorEnabled = true;

	public function __construct(
		Articles       $articles,
		Categories     $categories,
		Images         $images,
		ArticlesFacade $articlesFacade
	)
	{
		$this->articlesService   = $articles;
		$this->categoriesService = $categories;
		$this->imagesService     = $images;
		$this->articlesFacade    = $articlesFacade;
		$this->limit             = (int) BlogConfig::load('articlesListLimit', 8);

		$this->monitor(FrontPresenter::class, function(FrontPresenter $presenter) {
			$this->activeNavigation = $presenter->getActiveNavigation();
			$this->cache            = new Cache($this->cacheStorage, Articles::CACHE_NAMESPACE);

			$this->sort       = $this->activeNavigation->componentParams['sort'] ?? 'publishUp';
			$this->showLocked = (bool) ($this->activeNavigation->componentParams['showLocked'] ?? false);

			$this->page = $this->getParameter('page', 1);
		});
	}

	public function render(): void
	{
		$this->template->url              = trim($this->template->baseUrl, '/');
		$this->template->articlesCount    = $this->getArticlesCount();
		$this->template->limit            = $this->limit;
		$this->template->start            = $this->start;
		$this->template->paginatorEnabled = $this->paginatorEnabled;
		$this->template->articles         = $this->getArticles();
		$this->template->category         = $this->getCategory();

		$ids                        = is_array($this->categories) ? $this->categories : [$this->categories];
		$this->template->categories = $this->categoriesService->getCategories($ids);

		$paginator = null;
		if ($this->paginatorEnabled) {
			$paginator = new Paginator;
			$paginator->setItemCount($this->getArticlesCount());
			$paginator->setItemsPerPage($this->limit);
			$paginator->setPage($this->page);
		}
		$this->template->paginator = $paginator;

		$this->template->render($this->getTemplateFile());
	}

	/**
	 * @param string|int $page
	 */
	public function handlePaginator($page): void
	{
		$page        = (int) $page;
		$this->page  = $page;
		$this->start = $page > 1 ? $this->limit * ($page - 1) : 0;
	}

	/*******************************************************************************************************************
	 * ============================== Get / Set
	 */

	/**
	 * Načtení článků které se zobrazí na stránce
	 *
	 * @return Article[]|null
	 * @throws Throwable
	 */
	public function getArticles(): ?array
	{
		if (!$this->articles) {
			$this->articles = $this->articlesService->getAll($this->categories, $this->start, $this->limit, $this->sort, $this->showLocked);
		}

		return $this->articles;
	}

	/**
	 * Vrací categorii popřípadě kategorie listovaných článků
	 *
	 * @return int[]|Category|null
	 */
	public function getCategory()
	{
		if (is_array($this->categories) && count($this->categories) === 1) {
			return $this->categoriesService->get((int) $this->categories[0]);
		}

		//TODO tohle vraci jen IDCKA dopsat metodu do modelove tridy Categories
		return $this->categories;
	}

	/**
	 * Ziskání počtu zveřejněných článků
	 */
	public function getArticlesCount(): int
	{
		if (!$this->articlesCount) {
			$this->articlesCount = $this->articlesService->getAllCount($this->categories, $this->showLocked);
		}

		return $this->articlesCount;
	}

	/**
	 * @param int[]|null $categories
	 */
	public function setCategories(?array $categories): void
	{
		$this->categories = $categories;
	}

	public function setLimit(int $limit): void
	{
		$this->limit = $limit;
	}

}
