<?php declare(strict_types = 1);

namespace Blog\Model;

use Blog\Model\Entities\Article;
use Blog\Model\Entities\Category;
use Blog\Model\Entities\Hit;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Images\ImagePipe;
use Core\Model\Lang\DefaultLang;
use Doctrine\Common\Collections\Criteria;
use Doctrine\ORM\NonUniqueResultException;
use Doctrine\ORM\Query\QueryException;
use Exception;
use Gallery\Model\Images;
use Core\Model\Entities\QueryBuilder;
use Nette\Caching\Cache;
use Nette\Utils\DateTime;
use Tags\Model\Entities\Tag;
use Throwable;

/**
 * @method Article|null getReference($id)
 * @method Article[]|null getAll()
 * @method Article|null get($id)
 */
class Articles extends BaseEntityService
{
	protected $entityClass = Article::class;

	protected DefaultLang $defaultLang;
	protected ImagePipe   $imagePipe;
	protected Images      $imagesService;

	public const CACHE_NAMESPACE = 'articles';

	public function __construct(
		ImagePipe   $imagePipe,
		Images      $images,
		DefaultLang $defaultLang
	)
	{
		$this->imagePipe     = $imagePipe;
		$this->imagesService = $images;
		$this->defaultLang   = $defaultLang;
	}

	private function getCache(): Cache
	{
		if ($this->cache === null) {
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);
		}

		return $this->cache;
	}

	/**
	 * @return Article[]
	 */
	public function getPublishedByIds(array $ids): array
	{
		$articles = $this->getPublishedArticlesQuery()
			->andWhere('a.id IN (:aId)')
			->setParameter('aId', $ids)
			->getQuery();

		return $articles->getResult();
	}

	/**
	 * @param int|string|null $limit
	 *
	 * @return Article[]
	 */
	public function getNextPublishedArticle(Article $article, $limit = 1): array
	{
		return $this->getPublishedArticlesQuery()
			->andWhere('a.publishUp > :articlePublishUp')
			->setParameter('articlePublishUp', $article->publishUp)
			->orderBy('a.publishUp', 'ASC')
			->setMaxResults((int) $limit)
			->getQuery()
			->enableResultCache(120)
			->getResult();
	}

	/**
	 * @param Article         $article
	 * @param int|string|null $limit
	 *
	 * @return Article[]
	 */
	public function getPrevPublishedArticle(Article $article, $limit = 1): array
	{
		return $this->getPublishedArticlesQuery()
			->andWhere('a.publishUp < :articlePublishUp')
			->setParameter('articlePublishUp', $article->publishUp)
			->orderBy('a.publishUp', 'DESC')
			->setMaxResults((int) $limit)
			->getQuery()
			->enableResultCache(120)
			->getResult();
	}

	/**
	 * @param int|string $id
	 *
	 * @throws NonUniqueResultException
	 */
	public function getArticle($id): ?Article
	{
		return $this->getEr()->createQueryBuilder('a')->addSelect('c')
			->where('a.id = :id')->setParameter('id', $id)
			->leftJoin('a.category', 'c')
			->getQuery()->getOneOrNullResult();
	}

	/**
	 * @throws NonUniqueResultException
	 */
	public function getFeatured(): ?Article
	{
		$article = $this->getPublishedArticlesQuery()->andWhere('a.featured IS NOT null')->setMaxResults(1)
			->getQuery()->getOneOrNullResult();

		if (!$article) {
			return null;
		}

		$article->getImage();

		return $article;
	}

	/**
	 * @return Article[]
	 */
	public function getAvailableArticles(): array
	{
		/** @var Article[] $tmp */
		$tmp = $this->getEr()->findAll();

		return $tmp;
	}

	public function publishedArticlesCriteria(): Criteria
	{
		$expr   = Criteria::expr();
		$now    = new DateTime;
		$locale = $this->defaultLang->locale;

		return Criteria::create()
			->andWhere(
				$expr->andX(
					$expr->andX(
						$expr->orX($expr->isNull('a.publishUp'), $expr->lte('a.publishUp', $now)),
						$expr->orX($expr->isNull('a.publishDown'), $expr->gte('a.publishDown', $now))
					),
					$expr->neq('a.publishUp', null),
					$expr->neq('a.title', null),
					$expr->neq('a.category', null),
					$expr->eq('a.isPublished', 1),
					$expr->eq('c.isPublished', 1),
					$expr->isNull('c.password'),
					$expr->orX(
						$expr->isNull('c.lang'),
						$expr->eq('c.lang', $locale),
						$expr->eq('c.lang', '')
					)
				)
			);
	}

	/**
	 * @param QueryBuilder|\Doctrine\ORM\QueryBuilder $qb
	 *
	 * @throws QueryException
	 */
	public function publishedArticlesSetQB($qb): void
	{
		$qb->leftJoin(Category::class, 'c', 'WITH', 'a.category = c.id')
			->addCriteria($this->publishedArticlesCriteria());
	}

	/**
	 * TODO udělat přes funkci publishedArticlesCriteria
	 */
	public function getPublishedArticlesQuery(bool $withLocked = false, array $skip = []): \Doctrine\ORM\QueryBuilder
	{
		$qb = $this->getEr()->createQueryBuilder('a')
			->andWhere('a.title IS NOT null')->andWhere('a.category IS NOT null')
			->andWhere('a.isPublished = 1')
			->leftJoin(Category::class, 'c', 'WITH', 'a.category = c.id')
			->andWhere('c.isPublished = 1')
			->andWhere('(c.lang IS null OR c.lang = :locale OR c.lang = \'\')')
			->setParameter('now', new DateTime())
			->setParameter('locale', $this->defaultLang->locale)
			->groupBy('a.id');

		$publish = ['(a.publishDown >= :now OR a.publishDown IS null)'];
		if (!in_array('publishUp', $skip, true)) {
			$publish[] = '(a.publishUp IS null OR a.publishUp <= :now)';
			$publish[] = 'a.publishUp IS NOT null';
		}

		$qb->andWhere(implode(' AND ', $publish));

		if (!$withLocked) {
			$qb->andWhere('c.password IS null');
		}

		return $qb;
	}

	/**
	 * @return Article[]
	 */
	public function getPublishedArticles(?array $categories = null): array
	{
		$qb = $this->getPublishedArticlesQuery();

		if ($categories) {
			$qb->andWhere('a.category IN (:categories)')->setParameter('categories', $categories);
		}

		return $qb->getQuery()->getResult();
	}

	/**
	 * @param int|string $articleId
	 */
	public function removeArticle($articleId): bool
	{
		if ($article = $this->getEr()->find($articleId)) {
			$this->em->remove($article);
			$this->em->flush();

			return true;
		}

		return false;
	}

	/**
	 * @param int[]|string[] $ids
	 */
	public function removeArticles(array $ids): array
	{
		$errors = [];
		foreach ($ids as $id) {
			if (!$this->removeArticle($id)) {
				$errors[] = $id;
			}
		}

		return $errors;
	}

	/**
	 * @param int $articleId
	 * @param int $featureTime
	 *
	 * @return bool
	 * @throws Exception
	 */
	public function setFeature($articleId, $featureTime): bool
	{
		/** @var Article|null $article */
		$article = $this->getEr()->find($articleId);
		if ($article) {
			if ($featureTime > 1) {
				$endTime = (new DateTime())->modify("+$featureTime minutes");
			} else {
				$endTime = $featureTime;
			}

			if (BlogConfig::load('onlyOneFeatured', true)) {
				$this->em->createQueryBuilder()->update(Article::class, 'a')->set('a.featured', 'null')
					->where('a.featured IS NOT null')
					->getQuery()->execute();
			}

			$article->setFeatured((string) $endTime);
			$this->em->persist($article);
			$this->em->flush();
			$this->getCache()->clean([Cache::TAGS => [self::CACHE_NAMESPACE . '/featured']]);

			return true;
		}

		return false;
	}

	public function checkFeatured(): void
	{
		// TODO rewrite
		//		$langs            = array_keys(Langs::$langs);
		//		$featuredArticles = [];
		//
		//		foreach ($langs as $v) {
		//			$featuredArticles[$v] = null;
		//		}
		//		foreach ($this->getEr()->findBy(['featured !=' => null]) as $article) {
		//			/** @var Article $article */
		//			$featuredArticles[$article->getCategory()->getLangId()] = $article;
		//		}
		//
		//		try {
		//			foreach ($featuredArticles as $lang => $featured) {
		//				if ($featured == null
		//					|| ($featured->getFeatured() != 1 && (new DateTime('now'))->getTimestamp() > DateTime::from($featured->getFeatured())->getTimestamp())
		//					|| ($featured->getFeatured() == 1)
		//				) {
		//					$lastArticle = $this->getPublishedArticlesQuery()->setParameter('locale', $lang)->orderBy('a.publishUp', 'DESC')->setMaxResults(1)->getQuery()->getSingleResult();
		//					if (!$featured || $lastArticle->getId() != $featured->getId()) {
		//						$this->setFeature($lastArticle->getId(), 1);
		//					}
		//				}
		//			}
		//		} catch (Exception $e) {
		//		}
	}

	/**
	 * @param int|string $categoryId
	 */
	public function getArticlesCountInCategoryByPublishing($categoryId): array
	{
		$arr = [
			0 => ['count' => 0, 'isPublished' => 0],
			1 => ['count' => 0, 'isPublished' => 1],
		];
		$qb  = $this->getEr()->createQueryBuilder('a', 'a.isPublished')->select('COUNT(a.id) as count, a.isPublished')
			->where('a.category = :category')->setParameter('category', $categoryId)
			->groupBy('a.isPublished');

		return (($qb->getQuery()->getResult() ?: []) + $arr);
	}

	/**
	 * @param int|string $articleId
	 *
	 * @return int[]|null
	 * @throws Throwable
	 */
	public function getRelated($articleId, ?int $limit = 1): ?array
	{
		try {
			$key = self::CACHE_NAMESPACE . '/related/' . $articleId;

			$ids = $this->getCache()->load($key, function(&$dep) use ($key, $articleId) {
				$dep     = [Cache::TAGS => [$key, self::CACHE_NAMESPACE], Cache::EXPIRE => '1 day'];
				$article = $this->getReference($articleId);

				$qb = $this->getPublishedArticlesQuery();

				$ids    = [];
				$result = $qb->select('a.id')
					->andWhere('a.publishUp > :from')
					->join(Tag::class, 't', 'WITH', 't.id IN (:tags)')
					->groupBy('a.id')
					->setParameter('tags', $article->getTagsId())
					->setParameter('from', DateTime::createFromFormat('Y-m-d', '2015-01-01'))
					->setMaxResults(200)->orderBy('a.publishUp', 'DESC')->getQuery()->getArrayResult();

				foreach ($result as $r) {
					$ids[] = $r['id'];
				}

				return $ids;
			});

			shuffle($ids);

			return array_slice($ids, 0, $limit);
		} catch (Exception $e) {
		}

		return null;
	}

	/**
	 * @param mixed $article
	 * @param mixed $link
	 */
	public function getPreparedText(&$article, $link): void
	{
		// TODO rewrite
		//		if ($article->fulltext) {
		//			$scripts = [];
		//
		//			$html = HtmlDomParser::str_get_html($article->fulltext);
		//			foreach ($html->find('.socialshare') as $v) {
		//				$src = (string) $v->{'data-src'};
		//				if (strpos($src, 'youtube.com') !== false) {
		//					preg_match("#(?<=v=)[a-zA-Z0-9-]+(?=&)|(?<=v\/)[^&\n]+(?=\?)|(?<=v=)[^&\n]+|(?<=youtu.be/)[^&\n]+#", $src, $matches);
		//
		//					if (isset($matches[0])) {
		//						$v->outertext = '<div class="iframe-wrap"><iframe src="https://www.youtube.com/embed/' . $matches[0] . '" class="youtube-iframe" frameborder="0"></iframe></div>';
		//					}
		//					else {
		//						$v->outertext = '';
		//					}
		//				} else if (strpos($src, 'vimeo.com') !== false) {
		//					$src          = explode('/', $src);
		//					$v->outertext = '<div class="iframe-wrap"><iframe src="https://player.vimeo.com/video/' . end($src) . '" frameborder="0" webkitallowfullscreen mozallowfullscreen allowfullscreen></iframe></div>';
		//				} else if (strpos($src, '.gifv') !== false || strpos($src, '.webm') !== false) {
		//					$src          = str_replace('.gifv', '.webm', $src);
		//					$v->outertext = '<div class="text-align: center;"><video style="max-width: 100%" preload="auto" autoplay="autoplay" loop="loop"><source src="' . $src . '" type="video/webm"></video></div>';
		//				} else if (strpos($src, '.gif') !== false) {
		//					$v->outertext = "<div style='text-align: center'><img src='{$src}'></div>";
		//				} else if (strpos($src, 'instagram.com') !== false) {
		//					$scripts['instagram'] = "<script async defer='defer' src='//platform.instagram.com/en_US/embeds.js'></script>";
		//					$v->outertext         = "<blockquote class='instagram-media' data-instgrm-captioned data-instgrm-version='4'>"
		//						. "<a href='{$src}'></a>"
		//						. "</blockquote>";
		//				} else if (strpos($src, 'facebook.com') !== false) {
		//					$v->outertext = "<div class='fb-post' data-href='{$src}' data-width='500' data-show-text='true'></div>";
		//				} else if (strpos($src, 'twitter.com') !== false) {
		//					$scripts['twitter'] = "<script async src='https://platform.twitter.com/widgets.js' charset='utf-8'></script>";
		//					$v->outertext       = "<blockquote class='twitter-tweet' data-lang='cs'><a href='{$src}'></a></blockquote>";
		//				} else if (strpos($src, 'tumblr.com') !== false) {
		//					$scripts['tumblr'] = '<script async src="https://assets.tumblr.com/post.js"></script>';
		//					$v->outertext      = '<div class="tumblr-post" data-href="https://embed.tumblr.com/embed/post/0FTBaMwb9fSe1ArdSIeGBg/171646778196" data-did="4840232ac009c4f4135503e8c14006b4120200a4"></div>';
		//				} else if (strpos($src, 'giphy.com') !== false) {
		//					$arr          = explode('-', $src);
		//					$v->outertext = "<div class='iframe-wrap'><iframe src='https://giphy.com/embed/" . array_pop($arr) . "?video=0' frameBorder='0' class='giphy-embed' allowFullScreen></iframe></div>";
		//				} else if (strpos($src, 'imgur.com') !== false) {
		//					$arr  = explode('/', $src);
		//					$code = array_pop($arr);
		//					if (end($arr) == 'gallery') {
		//						$code = 'a/' . $code;
		//					}
		//					$scripts['imgur'] = "<script async src='//s.imgur.com/min/embed.js' charset='utf-8'></script>";
		//					$v->outertext     = "<div style='text-align: center;'><blockquote class='imgur-embed-pub' data-id='" . $code . "'></blockquote></div>";
		//				} else if (strpos($src, 'spotify.com') !== false) {
		//					$v->outertext = '<iframe src="https://open.spotify.com/embed?uri=' . $src . '" width="300" height="380" frameborder="0" allowtransparency="true" allow="encrypted-media"></iframe>';
		//				}
		//			}
		//
		//			foreach ($html->find('img') as $img) {
		//				if (strpos($img->src, 'images') === 0) {
		//					$img->src = '/' . $img->src;
		//				}
		//			}
		//
		//			$html = implode('', $scripts) . $html;
		//
		//			$article->preparedFulltext = (string) $html;
		//		} else {
		//			$article->preparedFulltext = $article->fulltext;
		//		}
		//
		//		try {
		//			$relatedKey = "/<div class=\"relatedarticle\">(.*?)<\/div>/";
		//			$text       = $article->getPreparedFulltext();
		//			$count      = preg_match_all($relatedKey, $text);
		//
		//			if (empty($article->getTagsId()) || $count == 0) {
		//				$text = preg_replace($relatedKey, '', $text);
		//			} else {
		//				$relatedArticles = $this->getRelated($article->getId(), $count);
		//
		//				self::$i = 0;
		//				$text    = preg_replace_callback($relatedKey, function($match) use ($relatedArticles, $link) {
		//					if (!isset($relatedArticles[self::$i])) {
		//						return '';
		//					}
		//					$articleId = $relatedArticles[self::$i];
		//					self::$i++;
		//					$keyA = self::CACHE_NAMESPACE . '/article/' . $articleId;
		//					$key  = self::CACHE_NAMESPACE . '/relatedInArticleText/' . $articleId;
		//
		//					return $this->getCache()->load($key, function(&$dep) use ($key, $keyA, $articleId, $link) {
		//						$dep = [Cache::TAGS => [$key, self::CACHE_NAMESPACE, $keyA], Cache::EXPIRE => '1 day'];
		//
		//						$a   = $this->getArticle($articleId);
		//						$img = $a->getImageFile() ? $this->imagePipe->request($a->getImageFile(), '171x110', 'fill') : '#';
		//
		//						return Html::el('div class=related-article')
		//							->addHtml(Html::el('a', ['href' => $link(':Blog:Front:Articles:detail', ['id' => $a->getId()])])
		//								->addHtml(Html::el('span class=img')->addHtml(Html::el('img', ['src' => $img])))
		//								->addHtml(Html::el('span class=text')
		//									->addHtml(Html::el('span class=date')->setText($a->publishUp->format('j. n. Y')))
		//									->addHtml(Html::el('span class=t')->setText($a->title))));
		//					});
		//				}, $text);
		//			}
		//
		//			$imageKey = "/<div class=\"albumimages\" data-src=\"(\d*)\">(.*?)<\/div>/";
		//			$text     = preg_replace_callback($imageKey, function($match) {
		//				if (!isset($match[1]) || !($img = $this->imagesService->get($match[1]))) {
		//					return '';
		//				}
		//
		//				return Html::el('div class=album-image')
		//					->addHtml('<img src="' . $img->getFilePath() . '">')
		//					->addHtml(Html::el('em class=album-image-description')->setText($img->description));
		//			}, $text);
		//
		//			$article->preparedFulltext = $text;
		//		} catch (Exception $e) {
		//		}
		//
		//		$article->preparedFulltext = str_replace('[timestamp]', time(), $article->preparedFulltext);
		//
		//		return $article->preparedFulltext;
	}

	public function getMostRead(int $limit = 5): array
	{
		$qb = $this->em->getRepository(Hit::class)->createQueryBuilder('h')
			->select('IDENTITY(h.article) as articleId, SUM(h.shows) as hits')
			->join(Article::class, 'a', 'WITH', 'a.id = h.article')
			->andWhere('h.date >= :dateLimit')->setParameter('dateLimit', (new DateTime)->modify('-3 days'))
			->groupBy('h.article')
			->orderBy('hits', 'DESC');

		$this->publishedArticlesSetQB($qb);
		$qb->setMaxResults($limit);

		return $qb->getQuery()->getArrayResult();
	}

	public function searchFulltext(string $word, int $offset = 0, int $limit = 20): array
	{
		return $this->getPublishedArticlesQuery()->andWhere('a.title LIKE :word OR a.fulltext LIKE :word')
			->setParameter('word', "%$word%")
			->setFirstResult($offset)->setMaxResults($limit)
			->orderBy('a.publishUp', 'DESC')
			->getQuery()->getArrayResult();
	}
}
