<?php declare(strict_types = 1);

namespace Blog\Model\Entities;

use Blog\FrontModule\Model\Articles as FrontArticles;
use Blog\Model\Articles;
use Blog\Model\BlogCache;
use Blog\Model\Categories;
use Core\Model\Sites;
use Doctrine;
use Doctrine\ORM\Event\LifecycleEventArgs;
use Doctrine\ORM\Mapping as ORM;
use Navigations\Model\Navigations;
use Nette\Caching\Cache;
use Nette\Caching\Storage;
use Nette\Http\Session;
use Nette\SmartObject;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class ArticleListener implements EventSubscriberInterface
{
	use SmartObject;

	protected static int    $i = 0;
	protected Cache         $cache;
	protected Cache         $navigationCache;
	protected Cache         $categoriesCache;
	protected BlogCache     $blogCache;
	protected Session       $session;
	protected FrontArticles $frontArticles;
	protected Sites         $sites;

	public function __construct(
		Storage       $cacheStorage,
		Session       $session,
		BlogCache     $blogCache,
		FrontArticles $frontArticles,
		Sites         $sites
	)
	{
		$this->session         = $session;
		$this->cache           = new Cache($cacheStorage, Articles::CACHE_NAMESPACE);
		$this->categoriesCache = new Cache($cacheStorage, Categories::CACHE_NAMESPACE);
		$this->navigationCache = new Cache($cacheStorage, Navigations::CACHE_NAMESPACE);
		$this->blogCache       = $blogCache;
		$this->frontArticles   = $frontArticles;
		$this->sites           = $sites;
	}

	public static function getSubscribedEvents(): array
	{
		return [];
	}

	/**
	 * @ORM\PreRemove
	 */
	public function onPreRemove(Article $article, Doctrine\ORM\Event\LifecycleEventArgs $event): void
	{
		$sessionSection = $this->session->getSection('unsavedArticles');
		unset($sessionSection->{'article_' . $article->getId()});
	}

	/**
	 * @ORM\PostPersist
	 * @ORM\PostUpdate
	 * @ORM\PostRemove
	 */
	public function postHandler(Article $article, LifecycleEventArgs $event): void
	{
		foreach ($article->getTexts()->getKeys() as $l) {
			$this->cache->remove('article/' . $article->getId() . '/' . $l);
			$this->blogCache->getCache()->remove('article/' . $article->getId() . '/' . $l);
			$this->categoriesCache->remove('categories/' . $l);
		}

		$this->navigationCache->clean([Cache::TAGS => 'article/' . $article->getId()]);

		$changeSet = $event->getEntityManager()->getUnitOfWork()->getEntityChangeSet($article);
		if (isset($changeSet['featured'])) {
			$this->cache->clean([Cache::TAGS => [Articles::CACHE_NAMESPACE . '/featured']]);
		}

		$this->blogCache->getCache()->clean([
			Cache::Tags => ['getAll', 'author', 'featured'],
		]);

		$cRoot = $this->frontArticles->getCategoryRootForSite($this->sites->getCurrentSite()->getIdent());

		if ($cRoot) {
			foreach ($article->getTexts()->getKeys() as $l) {
				$this->blogCache->getCache()->remove('forNavigation_' . $cRoot . '_' . $l);
			}
		}

		$this->cache->clean([Cache::TAGS => ['articleNavs']]);
		$this->blogCache->getCache()->clean([Cache::TAGS => ['articleNavs']]);
	}
}
