<?php declare(strict_types = 1);

namespace Blog\Model\Navigation;

use Blog\FrontModule\Model\Articles;
use Blog\Model\BlogConfig;
use Blog\Model\Categories;
use Blog\Model\Entities\Article;
use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Dao\SiteMapUrl;
use Core\Model\Entities\ExtraField;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Navigations\Model\Helper\NavigationsHelper;
use Navigations\Model\Providers\INavigationItem;
use Nette\Caching\Cache;
use Nette\Utils\ArrayHash;
use Pages\FrontModule\Model\TemplatePages;
use Pages\Model\Entities\TemplatePage;

class Category extends BaseNavigation implements INavigationItem
{
	protected        $title                  = 'category';
	protected        $presenter              = 'Blog:Front:Categories';
	protected        $action                 = 'category';
	protected string $presenterArticleDetail = 'Blog:Front:Articles';
	protected string $actionArticleDetail    = 'detail';

	public const CACHE_ARTICLE = 'blogArticle';
	protected Categories        $categoriesService;
	protected Articles          $articlesService;
	protected NavigationsHelper $navigationsHelper;
	protected TemplatePages     $templatePagesService;

	public function __construct(Categories $categories, Articles $articles, NavigationsHelper $navigationsHelper, TemplatePages $templatePages)
	{
		$this->categoriesService    = $categories;
		$this->articlesService      = $articles;
		$this->navigationsHelper    = $navigationsHelper;
		$this->templatePagesService = $templatePages;
	}

	/**
	 * @param BaseForm $baseForm
	 *
	 * @return BaseContainer
	 */
	public function getFormContainer(BaseForm $baseForm)
	{
		$container = new BaseContainer;

		$categories = [];
		foreach ($this->categoriesService->getAll() as $category) {
			$categories[$category->getId()] = !$category->getText() ? ' ' : ($category->getText()->getTitle() ?: ' ');
		}

		$sort = [];
		foreach (['default', 'random'] as $v) {
			$sort[$v] = $this->t('blog.navigationBuilder.sortTypes.' . $v);
		}

		$container->addSelect('category', 'blog.navigationBuilder.category', $categories);
		$container->addSelect('showPublishedInFuture', 'blog.navigationBuilder.showPublishedInFuture', [
			0 => $this->t('default.form.no'),
			1 => $this->t('default.form.yes'),
		])->setDefaultValue(0);
		$container->addSelect('showLocked', 'blog.navigationBuilder.showLocked', [
			0 => $this->t('default.form.no'),
			1 => $this->t('default.form.yes'),
		])->setDefaultValue(0);
		$container->addSelect('sort', 'blog.navigationBuilder.sort', $sort);

		$templates = [null => ' '];
		foreach ($this->templatePagesService->getEr()->findBy([]) as $t) {
			/** @var TemplatePage $t */
			$templates[$t->getId()] = $t->title;
		}
		$container->addSelect('tpCategory', $this->t('blog.navigationBuilder.templatePageCategory'), $templates)
			->setTranslator(null);
		$container->addSelect('tpDetail', $this->t('blog.navigationBuilder.templatePageDetail'), $templates)
			->setTranslator(null);

		return $container;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
	}

	public function updateCacheDep(array &$cache, $urlParams)
	{
		if ($urlParams['id']) {
			$cache[Cache::TAGS][] = 'article/' . $urlParams['id'];
			$cache[Cache::TAGS][] = 'articles/';
		}

		parent::updateCacheDep($cache, $urlParams);
	}

	/*******************************************************************************************************************
	 * ===========================  Route
	 */

	public function presenterActionCheck($presenter, $action)
	{
		if ($this->presenter == $presenter && $this->action == $action) {
			return 1;
		}
		if ($this->presenterArticleDetail == $presenter && $this->actionArticleDetail == $action) {
			return 1;
		}

		return false;
	}

	/**
	 * @param DaoNavigationItem $navigation
	 * @param array             $urlParams
	 */
	public function routerIn($navigation, $urlParams = [])
	{
		$params = explode('/', $urlParams['path']);

		if (isset($urlParams['path']) && count($params) > 1) {
			$id = explode('-', end($params), 2);

			$return = [
				'presenter' => $this->presenterArticleDetail,
				'action'    => $this->actionArticleDetail,
			];

			$articleId = null;
			if (BlogConfig::load('removeArticleIdFromUrl', false)) {
				$articleId = $this->articlesService->getPublishedIdByAlias(end($params), $urlParams['locale'])['id'] ?? null;
			}

			if (!$articleId) {
				if (!is_numeric($id[0])) {
					$articleId = $this->articlesService->getPublishedIdByAlias(end($params), $urlParams['locale'])['id'] ?? null;
				} else {
					$articleId = $id[0];
				}
			}

			if ($articleId) {
				$return['id'] = $articleId;

				return $return;
			}
		}

		return [
			'presenter' => $this->presenter,
			'action'    => $this->action,
			'id'        => $navigation->componentParams['category'],
		];
	}

	public function routerOut(DaoNavigationItem $navigation, &$urlParams): ?string
	{
		if (isset($urlParams['action']) && $urlParams['action'] == $this->actionArticleDetail && isset($urlParams['id'])) {
			if (BlogConfig::load('allowArticleFullUrlField', false)) {
				$r = $this->categoriesService->em->getRepository(ExtraField::class)->createQueryBuilder('ef')
					->select('ef.value')
					->where('ef.sectionName = :sn')
					->andWhere('ef.sectionKey = :sk')
					->andWhere('ef.lang = :lang')
					->andWhere('ef.key = :key')
					->setParameters([
						'sn'   => Article::EXTRA_FIELD_SECTION,
						'key'  => 'fullUrl',
						'sk'   => $urlParams['id'],
						'lang' => $urlParams['locale'],
					])->getQuery()->getScalarResult()[0]['value'] ?? '';

				if ($r) {
					return '/' . $r;
				}
			}

			$article = $this->articlesService->getForNavigation((int) $urlParams['id'], $urlParams['locale'] ?: $navigation->getLang());

			if ($navigation->componentType === 'blog.navigation.category' && (int) $navigation->componentParams['category'] === (int) $article['category']) {
				$articleNav = $navigation;
			} else {
				return null;
			}

			if ($article) {
				$articleUrl = $article['alias'];

				if (!BlogConfig::load('removeArticleIdFromUrl', false)) {
					$articleUrl = $article['id'] . '-' . $articleUrl;
				}

				return ($articleNav->isHomepage ? '/' : '/' . $articleNav->alias) . '/' . $articleUrl;
			}
		}

		if (isset($urlParams['id'])) {
			if ($navigation->componentType === 'blog.navigation.category' && (int) $navigation->componentParams['category'] === (int) $urlParams['id']) {
				return $navigation->isHomepage ? '/' : '/' . $navigation->alias;
			}

			return null;
		}

		return $navigation->isHomepage ? '/' : '/' . $navigation->alias;
	}

	public function getComponentChild(string $lang, $navigation = null, $siteMapData = false)
	{
		$arr = [];

		foreach ($this->articlesService->getAll([$navigation->componentParams['category']]) as $article) {
			$data = [
				'action'    => $this->actionArticleDetail,
				'id'        => $article->getId(),
				'presenter' => $this->presenterArticleDetail,
			];

			if ($siteMapData) {
				$siteMapUrl = new SiteMapUrl;
				if ($article->getModified()) {
					$siteMapUrl->setModified($article->getModified());
				}
				$siteMapUrl->setSeo($article->getSeo());

				if ($article->getGallery()) {
					foreach ($article->getGallery()->getImages() as $img) {
						$siteMapUrl->addImage($img);
					}
				}

				$data['siteMapData']['default'] = $siteMapUrl;
			}

			if ($navigation) {
				$data['activeNavigation'] = $navigation;
			}

			$arr[] = $data;
		}

		return $arr;
	}
}
