<?php declare(strict_types = 1);

namespace Blog\Model;

use Blog\Model\Entities\Article;
use Blog\Model\Entities\Opened;
use Core\Model\Helpers\BaseEntityService;
use Nette\Utils\DateTime;
use Users\Model\Entities\User;

class OpenedArticlesService extends BaseEntityService
{
	protected $entityClass = Opened::class;

	public function isArticleOpened(int $id): ?Opened
	{
		/** @var Opened|null $tmp */
		$tmp = $this->getEr()->findOneBy(['article' => $id]);

		return $tmp;
	}

	/**
	 * @param int[]|string[]|int|string $ids
	 *
	 * @return Opened[]
	 */
	public function isArticlesOpened($ids): array
	{
		/** @var Opened[] $arr */
		$arr = $this->getEr()->findBy(['article' => $ids]);

		return $arr;
	}

	public function openArticle(Article $article, User $user): void
	{
		$opened = $this->isArticleOpened($article->getId());
		if (!$opened) {
			$opened = new Opened($article, $user);
		}

		$opened->user         = $user;
		$opened->lastActivity = new DateTime;

		$this->em->persist($opened);
		$this->em->flush();
	}

	/**
	 * Zkontroluje aktivitu a vrátí aktivní článek
	 */
	public function checkArticle(int $id): ?Opened
	{
		$date = (new DateTime)->modify('-5 minutes');

		$opened = $this->isArticleOpened($id);
		if ($opened && $opened->lastActivity < $date) {
			$this->em->remove($opened);
			$this->em->flush();
		}

		return $opened;
	}

	/**
	 * Zkontroluje poslední aktivitu a vrátí pole otevřených článků
	 *
	 * @param int[]|string[] $ids
	 *
	 * @return Opened[]
	 */
	public function checkArticles(array $ids): array
	{
		$date = (new DateTime)->modify('-5 minutes');

		$opened = $this->isArticlesOpened($ids);
		foreach ($opened as $open) {
			if ($open->lastActivity < $date) {
				$this->em->remove($open);
			}
		}
		if ($opened) {
			$this->em->flush();
		}

		return $opened;
	}

	/**
	 * @param int|string $id
	 */
	public function closeArticle($id): void
	{
		$opened = $this->isArticlesOpened($id);
		foreach ($opened as $open) {
			$this->em->remove($open);
		}
		if ($opened) {
			$this->em->flush();
		}
	}

}
