<?php declare(strict_types = 1);

namespace Blog\Model\Subscribers;

use Blog\AdminModule\Model\Articles;
use Blog\Model\BlogCache;
use Blog\Model\BlogConfig;
use Blog\Model\Entities\Article;
use Blog\Model\Entities\ArticleInCategory;
use Blog\Model\Entities\ArticleText;
use Blog\Model\Entities\Category;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\InstagramMediaEvent;
use Core\Model\Lang\Langs;
use DynamicModule\Model\Services\MultihubInstagramService;
use Gallery\Model\Entities\Album;
use Gallery\Model\Entities\Image;
use Nette\Utils\DateTime;
use Nette\Utils\FileSystem;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Users\Model\Entities\User;

class InstagramSubscriber implements EventSubscriberInterface
{
	protected EntityManagerDecorator $em;
	protected Articles               $articles;
	protected Langs                  $langsService;
	protected BlogCache              $blogCache;

	public function __construct(
		EntityManagerDecorator $entityManager,
		Articles               $articles,
		Langs                  $langsService,
		BlogCache              $blogCache
	)
	{
		$this->em           = $entityManager;
		$this->articles     = $articles;
		$this->langsService = $langsService;
		$this->blogCache    = $blogCache;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'dynamicModule.instagramMedia::created' => ['newMedia', 0],
			'dynamicModule.instagramMedia::updated' => ['updateMedia', 0],
		];
	}

	public function newMedia(InstagramMediaEvent $event): void
	{
		if (
			!class_exists('DynamicModule\Model\Services\MultihubInstagramService')
			|| !BlogConfig::load('instagramSubscriber.enabled')
		) {
			return;
		}


		$config = BlogConfig::load('instagramSubscriber.profileIdMap.' . $event->profileId);
		if (!$config) {
			return;
		}

		/** @var User $author */
		$author = $this->em->getReference(User::class, $config['authorId']);

		/** @var Category $category */
		$category  = $this->em->getReference(Category::class, $config['categoryId']);
		$startFrom = $config['startFrom'] ?? null;

		$title   = MultihubInstagramService::parseValue('caption:firstLine', $event->mediaData);
		$created = MultihubInstagramService::parseValue('timestamp', $event->mediaData);

		if ($startFrom && $created < $startFrom) {
			return;
		}

		$instagramImageUrl = MultihubInstagramService::parseValue('mediaUrl', $event->mediaData);
		$mediaId           = MultihubInstagramService::parseValue('mediaId', $event->mediaData);

		$article = new Article();
		$article->setParam('instagramMediaId', $mediaId);
		$this->em->persist($article);

		$articleInCategory = new ArticleInCategory($article, $category);
		$this->em->persist($articleInCategory);

		$article->categories->add($articleInCategory);

		$article->publishUp = DateTime::createFromFormat('Y-m-d H:i:s', $created) ?: new DateTime();
		$this->em->persist($article);

		foreach ($this->langsService->getLangs(false) as $lang) {
			$articleText              = new ArticleText($article, (string) $title, $lang->getTag());
			$articleText->fulltext    = '<p>' . MultihubInstagramService::parseValue('caption:skipFirstLine', $event->mediaData) . '</p>';
			$articleText->createdBy   = $author;
			$articleText->isPublished = 1;
			$this->em->persist($articleText);
		}

		$this->em->flush();

		if ($instagramImageUrl) {
			$album = new Album(UPLOADS_PATH . '/gallery');
			$this->em->persist($album);
			$this->em->flush();

			$instagramExtension = 'jpg';
			$imagePath          = UPLOADS_DIR . '/gallery/' . $album->getId() . '/' . $mediaId . '.' . $instagramExtension;

			FileSystem::copy($instagramImageUrl, $imagePath);

			$image = new Image($album, $mediaId . '.' . $instagramExtension);
			$image->setAlbum($album);
			$this->em->persist($image);

			$article->gallery = $album;
			$this->em->persist($article);
			$this->em->flush();
		}

		foreach ($this->langsService->getLangs(false) as $lang) {
			$this->blogCache->getCache()->remove('article/' . $article->getId() . '/' . $lang->getTag());
		}
	}

	public function updateMedia(InstagramMediaEvent $event): void
	{
		if (
			!class_exists('DynamicModule\Model\Services\MultihubInstagramService')
			|| !BlogConfig::load('instagramSubscriber.enabled')
		) {
			return;
		}


		$config = BlogConfig::load('instagramSubscriber.profileIdMap.' . $event->profileId);

		if (!$config) {
			return;
		}

		$startFrom = $config['startFrom'] ?? null;

		$title   = MultihubInstagramService::parseValue('caption:firstLine', $event->mediaData);
		$created = MultihubInstagramService::parseValue('timestamp', $event->mediaData);

		if ($startFrom && $created < $startFrom) {
			return;
		}

		$instagramImageUrl = MultihubInstagramService::parseValue('mediaUrl', $event->mediaData);
		$mediaId           = MultihubInstagramService::parseValue('mediaId', $event->mediaData);

		/** @var User $author */
		$author = $this->em->getReference(User::class, $config['authorId']);

		foreach ($this->em->getRepository(Article::class)->createQueryBuilder('a')
			         ->where('a.params LIKE :mediaId')
			         ->setParameter('mediaId', '%' . $mediaId . '%')
			         ->getQuery()->getResult() as $article) {
			/** @var Article $article */
			if ($article->getParam('instagramMediaId') !== $mediaId) {
				continue;
			}

			foreach ($this->langsService->getLangs(false) as $lang) {
				$articleText = $article->getText($lang->getTag());

				if (!$articleText) {
					$articleText = new ArticleText($article, (string) $title, $lang->getTag());
				}

				$articleText->fulltext    = '<p>' . MultihubInstagramService::parseValue('caption:skipFirstLine', $event->mediaData) . '</p>';
				$articleText->createdBy   = $author;
				$articleText->isPublished = 1;
				$this->em->persist($articleText);
			}

			if ($instagramImageUrl) {
				$album = $article->getGallery();

				if (!$album) {
					$album = new Album(UPLOADS_PATH . '/gallery');
					$this->em->persist($album);

					$article->gallery = $album;
				}

				$igImage            = null;
				$instagramExtension = 'jpg';
				foreach ($album->getImages() as $v) {
					if ($v->getFilename() === $mediaId . '.' . $instagramExtension) {
						$igImage = $v;
						break;
					}
				}

				if ($igImage) {
					if (!file_exists($igImage->getFile())) {
						FileSystem::copy($instagramImageUrl, $igImage->getFile());
						var_dump($instagramImageUrl);
						var_dump($igImage->getFile());
					}
				} else {
					$imagePath = UPLOADS_DIR . '/gallery/' . $album->getId() . '/' . $mediaId . '.' . $instagramExtension;

					FileSystem::copy($instagramImageUrl, $imagePath);

					$image = new Image($album, $mediaId . '.' . $instagramExtension);
					$image->setAlbum($album);
					$this->em->persist($image);
				}
			}

			$this->em->persist($article);
			$this->em->flush();
		}
	}
}
