<?php declare(strict_types = 1);

namespace Blog\AdminModule\Components\Category;

use Blog\Model\Categories;
use Blog\Model\Entities\Category;
use Blog\Model\Entities\CategoryText;
use Blog\Model\UI;
use Core\AdminModule\Model\Sites;
use Core\Model\UI\AdminPresenter;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Exception;
use InvalidArgumentException;
use Nette\Application\BadRequestException;
use Nette\Utils\ArrayHash;
use Nette\Utils\Strings;
use Nette\Utils\Validators;

class CategoryForm extends BaseControl
{
	/** @var int|null @persistent */
	public ?int      $categoryId = null;
	public ?string   $siteIdent  = null;
	public ?Category $category   = null;

	public function __construct(
		protected UI\Category\Renderer $categoryRenderer,
		protected Categories           $categoriesService,
		protected Sites                $sitesService,
	)
	{
		$this->monitor(AdminPresenter::class, function(AdminPresenter $presenter) {
			$httpRequest = $presenter->getHttpRequest();

			$site = $httpRequest->getPost('site') ?: $this['form']->getComponent('site')->getValue();
			$this->loadParentItems($site);

			$this['form']->getComponent('site')
				->setHtmlAttribute('data-link-on-change', $this->link('loadParents!', ['__val__']));
		});
	}

	public function render(): void
	{
		$this->template->thisForm = $this['form'];
		$this->template->renderer = $this['form']->getRenderer();
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleLoadParents(string $site): void
	{
		$presenter = $this->presenter;

		try {
			$this->loadParentItems($site);
			$presenter->flashMessageSuccess('default.loaded');
		} catch (Exception) {
			$presenter->flashMessageDanger('default.error');
		}

		$this->redrawControl('parent');
		$presenter->redrawControl('flashes');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$layouts = [];
		foreach ($this->categoryRenderer->getLayouts() as $k => $v) {
			$layouts[$k] = $v->name;
		}

		$form->addText('title', 'blog.categoryForm.title')
			->setMaxLength(255)
			->setIsMultilanguage();
		$form->addText('alias', 'blog.categoryForm.alias')
			->setMaxLength(255)
			->setIsMultilanguage();
		$form->addSelect('layout', 'blog.categoryForm.layout', $layouts);
		$form->addText('password', 'blog.categoryForm.password')
			->setNullable();
		$form->addBool('isPublished', 'blog.categoryForm.isPublished')
			->setDefaultValue(1);

		$sites = $this->sitesService->getOptionsForSelect();
		if (count($sites) > 1) {
			$form->addSelect('site', 'blog.categoryForm.site', $sites)
				->setRequired();
		} else {
			$form->addHidden('site', array_values($sites)[0]);
		}
		$form->getComponent('site')->setDefaultValue($this->siteIdent ?: array_values($sites)[0]);

		$form->addSelect('parent', $this->t('default.parent'), [])->setTranslator(null);

		$form->addEditor('text', 'blog.category.text')
			->setIsMultilanguage();

		$form->addSaveCancelControl();

		$form->onValidate[] = $this->formValidate(...);
		$form->onSuccess[]  = $this->formSuccess(...);

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values): void
	{
		$nameFilled = false;
		foreach ($values->title as $v) {
			if ($v) {
				$nameFilled = true;
			}
		}

		if (!$nameFilled) {
			$form->addError('blog.categoryForm.categoryTitleMissing');
		}

		foreach ($values->alias as $v) {
			if (!Validators::isNone($v) && Strings::webalize($v) !== $v) {
				$form->addError('blog.categoryForm.errors.aliasNotWebalized');
			}
		}

		if ($form->hasErrors()) {
			$this->redrawControl('form');
		}
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$langValues = $form->convertMultilangValuesToArray();
			/** @var CategoryText[] $texts */
			$texts = [];

			if ($this->categoryId) {
				$category = $this->categoriesService->get($this->categoryId);
				$texts    = $category->getTexts()->toArray();
			} else {
				$category = new Category();
			}

			foreach ($langValues as $l => $v) {
				if (!isset($texts[$l]) && $v['title']) {
					$texts[$l] = new CategoryText($category, $v['title'], $l);
				}

				if ($v['title'] == '' || $texts[$l]->getTitle() == '') {
					if ($texts[$l]) {
						$this->em->remove($texts[$l]);
					}

					unset($texts[$l]);
					continue;
				}

				$texts[$l]->setTitle($v['title'])
					->setAlias($v['alias'] ?: $v['title']);
				$texts[$l]->text = $v['text'];

				$this->em->persist($texts[$l]);
			}

			$category->setTexts($texts);
			$category->isPublished = $values->isPublished;
			$category->setPassword($values->password);
			$category->setLayout($values->layout);

			if ($values->parent) {
				$category->parent = $this->categoriesService->getReference($values->parent);
			}

			$this->em->persist($category);
			$this->em->flush();
			$form->addCustomData('categoryId', $category->getId());
			$form->addCustomData('siteIdent', $values->site);
			$this->presenter->flashMessageSuccess('default.saved');
		} catch (Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	public function setCategory(int $id): void
	{
		$this->category = $this->categoriesService->get($id);

		if (!$this->category) {
			throw new BadRequestException;
		}

		$form     = $this['form'];
		$category = $this->category;
		$d        = [
			'password'    => $category->password,
			'isPublished' => $category->isPublished,
		];

		foreach ($category->getTexts() as $l => $text) {
			$d['title'][$l] = $text->getTitle();
			$d['alias'][$l] = $text->getAlias();
			$d['text'][$l]  = $text->text;
		}

		if ($category->parent && array_key_exists(
				$category->parent->getId(),
				$form->getComponent('parent')
					->getItems(),
			)) {
			$d['parent'] = $category->parent->getId();
		}

		if (array_key_exists($category->layout, $form->getComponent('layout')->getItems())) {
			$d['layout'] = $category->layout;
		}

		if ($this->siteIdent) {
			$d['site'] = $this->siteIdent;
		}

		$form->setDefaults($d);
	}

	protected function loadParentItems(string $site): void
	{
		$rootId = $this->categoriesService->getEr()->createQueryBuilder('c')
			->select('c.id')
			->innerJoin('c.texts', 'ct')
			->where('c.lvl = :lvl')
			->andWhere('ct.alias = :alias')
			->setParameters([
				'lvl'   => 0,
				'alias' => $site,
			])->getQuery()->setMaxResults(1)->getArrayResult()[0]['id'] ?? null;

		if (!$rootId) {
			throw new InvalidArgumentException;
		}

		$categories = [$rootId => ' '];
		foreach ($this->em->getRepository(Category::class)->createQueryBuilder('c')
			         ->select('c.id, ct.title, c.lvl')
			         ->leftJoin('c.texts', 'ct', 'WITH', 'ct.lang = :lang')
			         ->where('c.lvl > 0')->andWhere('ct.title != \'\'')
			         ->andWhere('c.root = :root')
			         ->setParameters([
				         'lang' => $this->translator->getLocale(),
				         'root' => $rootId,
			         ])
			         ->orderBy('c.root')->addOrderBy('c.lft')
			         ->getQuery()->getResult() as $c) {
			$name = ' ' . $c['title'];
			for ($i = 1; $i < $c['lvl']; $i++) {
				$name = '---' . $name;
			}
			$categories[$c['id']] = trim($name);
		}

		$this['form']->getComponent('parent')->setItems($categories);
	}
}
