<?php declare(strict_types = 1);

namespace Blog\FrontModule\Presenters;

use Blog\FrontModule\Components\Article\ArticleDetail;
use Blog\FrontModule\Components\Article\IArticleDetailFactory;
use Blog\FrontModule\Model\Articles as FrontArticles;
use Blog\FrontModule\Model\Dao;
use Blog\Model\ArticleHits;
use Blog\Model\BlogConfig;
use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Helpers\Strings;
use Doctrine\ORM\NonUniqueResultException;
use Nette\Application\AbortException;
use Nette\Application\BadRequestException;
use Nette\Application\UI\InvalidLinkException;
use Pages\FrontModule\Model\TemplatePages;
use Pages\Model\Paths;
use Spatie\SchemaOrg\Schema;

class ArticlesPresenter extends BasePresenter
{
	public ?Dao\Article $article = null;

	public function __construct(
		protected ArticleHits   $articleHitsService,
		protected TemplatePages $templatePagesService,
		protected Paths         $pathsService,
		protected FrontArticles $frontArticles,
	)
	{
		parent::__construct();
	}

	/*******************************************************************************************************************
	 * ==================================   Handle
	 */

	/*******************************************************************************************************************
	 * ==================================   Renders
	 */

	public function renderDetail(): void
	{
		$this['meta']
			->setMeta('title', $this->article->title)
			->setMeta('og:title', $this->article->title, 'property')
			->setMeta('og:type', 'article', 'property')
			->setMeta('og:description', $this->article->introtext, 'property')
			->setMeta('og:url', $this->link('//detail', $this->article->getId()), 'property');

		if ($this->article->getGallery() && $this->article->getGallery()->getCover()) {
			$gallery = $this->article->getGallery();

			$this['meta']
				->setMeta('og:image', rtrim($this->baseUrl, '/') . $gallery->getCover()->getFilePath(), 'property')
				->setMeta('image', rtrim($this->baseUrl, '/') . $gallery->getCover()->getFilePath());
		}
	}

	/*******************************************************************************************************************
	 * ==================================   Actions
	 */

	public function actionDetail(int $id): void
	{
		$this->article = $this->articlesDaoService->get($id);

		if (!$this->article) {
			$this->error($this->t('blog.article.notFound'));
		}

		$this['articleDetail']->setArticle($this->article);

		$article = $this->article;

		if (BlogConfig::load('allowRichSnippets')) {
			$text   = Strings::normalizeText($article->fulltext);
			$schema = Schema::article()
				->headline(Strings::truncate($article->title, 110))
				->genre($article->category->title)
				->wordCount(Strings::wordsCount($text))
				->url($this->link('//this'))
				->author(Schema::person()->name($article->createdBy->getName()))
				->publisher(
					Schema::organization()
						->name($this->sitesService->getCurrentSite()->getSiteName())
						->logo($this->baseUrl . ltrim($this->sitesService->getCurrentSite()->logo, '/')),
				)
				->datePublished($article->publishUp ?? $article->created)
				->description(Strings::normalizeText($article->introtext))
				->articleBody($text);

			if ($article->created) {
				$schema->dateCreated($article->created);
			}

			if ($article->modified) {
				$schema->dateModified($article->modified);
			}

			if ($article->getGallery() && $article->getGallery()->getCover()) {
				$schema->image(
					$this->imagePipe->request(
						$article->getGallery()->getCover()
							->getFilePath(),
						'1200x0',
						'fill',
					),
				);
			}

			$this['dataSchemaControl']->addSchema($schema);
		}

		$this->breadcrumbsAfter[] = new DaoNavigationItem([
			'title' => $article->title,
			'link'  => $article->link,
		]);

		$seo = $article->getSeo();
		if (!isset($seo['title']) || $seo['title'] == '') {
			$seo['title'] = $article->title . ' $separator $siteName';
		}
		$this->seoDataFromEntity     = $seo;
		$this->template->articleLink = $this->link('//:Blog:Front:Articles:detail', [$article->getId()]);
		$this->template->article     = $article;
		$this->template->latest      = $this->frontArticles->getAll();

		$tpId = $this->getActiveNavigation()->componentParams['tpDetail'];
		if ($tpId) {
			$templatePage = $this->templatePagesService->get((int) $tpId);

			if ($templatePage) {
				$this->template->templateTextValues = $templatePage->getTexts();
				$this->template->setFile($this->pathsService->getTemplatePagesFile($templatePage->getTemplate()));
			}
		}
	}

	/**
	 * @param string $hash
	 *
	 * @throws NonUniqueResultException
	 * @throws BadRequestException
	 * @throws InvalidLinkException
	 */
	public function actionPreview(int|string $articleId, $hash): void
	{
		if (!$articleId || !$hash || md5($articleId . md5($articleId . 'article')) != $hash) {
			$this->error();
		}

		//		TODO rewrite
		//		$article = $this->article = $this->articlesService->getArticle($articleId);
		//		$this['articleDetail']->setArticle($this->article->getId());
		//		$this->articlesService->getPreparedText($article, [$this, 'link']);
		//		$this['title']->setTitle($article->title);
		//		$this['meta']->setMeta('robots', 'noindex,nofollow');
		//		$this->template->categoryAlias = Strings::webalize($article->category->title);
		//		$this->template->articleLink   = $this->link('//:Blog:Front:Articles:detail', [$article->getId()]);
		//		$this->template->related       = [];
	}

	/**
	 * @throws AbortException
	 */
	public function actionAddShow(int|string $id): void
	{
		if ($this->isAjax()) {
			$this->articleHitsService->addShow($id);
		}
		$this->terminate();
	}

	public function actionAddToFavourite(int $article, string $group): void
	{
		if ($this->getUser()
				->isLoggedIn() && $id = $this->favourites->addToFavourites(
				$article,
				$group,
				$this->getUser()
					->getId(),
			)) {
			$this->flashMessageSuccess('blogFront.favourite.articleAdded' . ucfirst($group));
			$this->payload->added       = true;
			$this->payload->favouriteId = $id;
			$this->payload->removeLink  = $this->link(':Blog:Front:Articles:removeFromFavourite', [
				'article' => $article,
				'group'   => $group,
			]);
			$this->payload->text        = $this->t('blogFront.recipeDetail.removeFromFavourites');
		} else {
			$this->flashMessageDanger('blogFront.favourite.articleAddError' . ucfirst($group));
			$this->payload->added = false;
		}

		$this->redrawControl('flashes');
	}

	public function actionRemoveFromFavourite(int $article, string $group): void
	{
		if ($this->getUser()->isLoggedIn() && $this->favourites->removeFromFavourites(
				$article,
				$group,
				$this->getUser()
					->getId(),
			)) {
			$this->flashMessageSuccess('blogFront.favourite.articleRemoved' . ucfirst($group));
			$this->payload->removed = true;
			$this->payload->addLink = $this->link(':Blog:Front:Articles:addToFavourite', [
				'article' => $article,
				'group'   => $group,
			]);
			$this->payload->text    = $this->t('blogFront.recipeDetail.toFavourites');
		} else {
			$this->flashMessageDanger('blogFront.favourite.articleRemoved' . ucfirst($group));
			$this->payload->removed = false;
		}

		$this->redrawControl('flashes');
	}

	/*******************************************************************************************************************
	 * ==================================   Components
	 */

	//	protected function createComponentArticlesList(IArticlesListFactory $factory)
	//	{
	//		$control = $factory->create();
	//		$control->setCategories([$this->article->category->getId()]);
	//
	//		return $control;
	//	}

	protected function createComponentArticleDetail(IArticleDetailFactory $factory,
	): ArticleDetail
	{
		$control = $factory->create();

		return $control;
	}
}
