<?php declare(strict_types = 1);

namespace Blog\Model\Entities;

use Blog\Model\Articles;
use Blog\Model\BlogCache;
use Blog\Model\Categories;
use Doctrine;
use Doctrine\ORM\Event\PostPersistEventArgs;
use Doctrine\ORM\Event\PostRemoveEventArgs;
use Doctrine\ORM\Event\PostUpdateEventArgs;
use Doctrine\ORM\Mapping as ORM;
use Navigations\Model\Navigations;
use Nette\Caching\Cache;
use Nette\Caching\Storage;
use Nette\Http\Session;
use Nette\SmartObject;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class ArticleListener implements EventSubscriberInterface
{
	use SmartObject;

	protected static int $i = 0;
	protected Cache      $cache;
	protected Cache      $navigationCache;
	protected Cache      $categoriesCache;

	public function __construct(
		protected Storage   $cacheStorage,
		protected Session   $session,
		protected BlogCache $blogCache,
	)
	{
		$this->cache           = new Cache($cacheStorage, Articles::CACHE_NAMESPACE);
		$this->categoriesCache = new Cache($cacheStorage, Categories::CACHE_NAMESPACE);
		$this->navigationCache = new Cache($cacheStorage, Navigations::CACHE_NAMESPACE);
	}

	public static function getSubscribedEvents(): array
	{
		return [];
	}

	#[ORM\PreRemove]
	public function onPreRemove(Article $article, Doctrine\ORM\Event\PreRemoveEventArgs $event): void
	{
		$sessionSection = $this->session->getSection('unsavedArticles');
		$sessionSection->remove('article_' . $article->getId());
	}

	#[ORM\PostPersist]
	#[ORM\PostUpdate]
	#[ORM\PostRemove]
	public function postHandler(Article $article, PostPersistEventArgs|PostUpdateEventArgs|PostRemoveEventArgs $event): void
	{
		foreach ($article->getTexts()->getKeys() as $l) {
			$this->cache->remove('article/' . $article->getId() . '/' . $l);
			$this->categoriesCache->remove('categories/' . $l);
		}

		$this->navigationCache->clean([Cache::Tags => 'article/' . $article->getId()]);

		$this->cache->clean([Cache::Tags => ['articleNavs']]);
		$changeSet = $event->getObjectManager()->getUnitOfWork()->getEntityChangeSet($article);
		if (isset($changeSet['featured'])) {
			$this->cache->clean([Cache::Tags => [Articles::CACHE_NAMESPACE . '/featured']]);
		}

		$this->blogCache->getCache()->clean([
			Cache::Tags => ['getAll', 'author', 'featured'],
		]);
	}

}
