<?php declare(strict_types = 1);

namespace Blog\Model\Entities;

use DateTimeInterface;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Nette\Utils\DateTime;
use Nette\Utils\Strings;

#[ORM\Table(name: 'blog__category_text')]
#[ORM\Index(columns: ['category_id', 'lang'], name: 'category')]
#[ORM\Entity]
class CategoryText
{
	#[ORM\Id]
	#[ORM\ManyToOne(targetEntity: Category::class, inversedBy: 'texts')]
	#[ORM\JoinColumn(name: 'category_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	public Category $category;

	#[ORM\Id]
	#[ORM\Column(name: 'lang', type: Types::STRING, length: 2, nullable: true)]
	protected ?string $lang = null;

	#[ORM\Column(name: 'title', type: Types::STRING, length: 255)]
	public ?string $title = null;

	#[ORM\Column(name: 'alias', type: Types::STRING, length: 255)]
	protected ?string $alias = null;

	#[ORM\Column(name: 'is_published', type: Types::SMALLINT, options: ['default' => 1])]
	public int $isPublished = 1;

	#[ORM\Column(name: '`text`', type: Types::TEXT, nullable: true)]
	public ?string $text = null;

	#[ORM\Column(name: 'created', type: Types::DATETIME_MUTABLE)]
	public ?DateTimeInterface $created = null;

	public function __construct(
		Category $category,
		string   $title,
		string   $lang,
	)
	{
		$this->category    = $category;
		$this->title       = $title;
		$this->lang        = $lang;
		$this->isPublished = 1;
		$this->created     = new DateTime;
	}

	public function getTitle(): ?string { return $this->title; }

	public function setTitle(string $title): self
	{
		$this->title = $title;
		if (!$this->alias) {
			$this->setAlias($title);
		}

		return $this;
	}

	public function getAlias(): ?string { return $this->alias ?: Strings::webalize($this->title); }

	public function setAlias(string $alias): self
	{
		$this->alias = Strings::webalize($alias);

		return $this;
	}
}
