<?php declare(strict_types = 1);

namespace Blog\Model\Navigation;

use Blog\FrontModule\Model\Articles;
use Blog\FrontModule\Model\Categories;
use Blog\Model\BlogConfig;
use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Dao\SiteMapUrl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Navigations\Model\Entities\Navigation;
use Navigations\Model\Helper\NavigationsHelper;
use Navigations\Model\NavigationConfig;
use Navigations\Model\Providers\INavigationItem;
use Nette\Caching\Cache;
use Nette\Utils\ArrayHash;
use Pages\FrontModule\Model\TemplatePages;
use Pages\Model\Entities\TemplatePage;

class Blog extends BaseNavigation implements INavigationItem
{
	protected        $title                  = 'blog';
	protected        $presenter              = 'Blog:Front:Default';
	protected        $action                 = 'default';
	protected string $presenterArticleDetail = 'Blog:Front:Articles';
	protected string $actionArticleDetail    = 'detail';

	final public const CACHE_ARTICLE = 'blogArticle';

	public function __construct(
		protected Categories    $categoriesService,
		protected Articles      $articlesService,
		protected TemplatePages $templatePagesService,
	)
	{
	}

	/**
	 * @inheritdoc
	 */
	public function getFormContainer(BaseForm $baseForm)
	{
		$container = new BaseContainer;

		$container->addCheckboxNestedList(
			'categories',
			'blog.navigationBuilder.selectCategory',
			$this->categoriesService->getFlatTree(),
		);

		$templates = [null => ' '];
		foreach ($this->templatePagesService->getEr()->findBy([]) as $t) {
			/** @var TemplatePage $t */
			$templates[$t->getId()] = $t->title;
		}
		$container->addSelect('tpCategory', $this->t('blog.navigationBuilder.templatePageCategory'), $templates)
			->setTranslator(null);
		$container->addSelect('tpDetail', $this->t('blog.navigationBuilder.templatePageDetail'), $templates)
			->setTranslator(null);
		$container->addBool('addCategoryAlias', 'blog.navigationBuilder.addCategoryAlias')
			->setDefaultValue(0);

		return $container;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
	}

	public function updateCacheDep(array &$cache, $urlParams)
	{
		if ($urlParams['id']) {
			$cache[Cache::Tags][] = 'article/' . $urlParams['id'];
			$cache[Cache::Tags][] = 'articles/';
		}

		parent::updateCacheDep($cache, $urlParams);
	}

	/*******************************************************************************************************************
	 * ===========================  Route
	 */

	public function presenterActionCheck($presenter, $action)
	{
		if ($this->presenter === $presenter && $this->action === $action) {
			return 1;
		}
		if ($this->presenterArticleDetail === $presenter && $this->actionArticleDetail === $action) {
			return 2;
		}

		return false;
	}

	/**
	 * @param Navigation|DaoNavigationItem $navigation
	 * @param array                        $urlParams
	 */
	public function routerIn($navigation, $urlParams = [])
	{
		$params = explode('/', (string) $urlParams['path']);

		if (isset($urlParams['path']) && count($params) > 1) {
			$endParam = end($params);
			$id       = explode('-', $endParam, 2);

			$return = [
				'presenter' => $this->presenterArticleDetail,
				'action'    => $this->actionArticleDetail,
			];

			$endParam = NavigationsHelper::removeUrlSuffix($endParam);

			$articleId = null;
			if (BlogConfig::load('removeArticleIdFromUrl', false)) {
				$articleId = $this->articlesService->getPublishedIdByAlias(
					$endParam,
					$urlParams['locale'],
				)['id'] ?? null;
			}

			if (!$articleId) {
				if (!is_numeric($id[0])) {
					$articleId = $this->articlesService->getPublishedIdByAlias(
						$endParam,
						$urlParams['locale'],
					)['id'] ?? null;
				} else {
					$articleId = $id[0];
				}
			}

			if ($articleId) {
				$return['id'] = $articleId;

				return $return;
			}
		}

		return [
			'presenter' => $this->presenter,
			'action'    => $this->action,
			'id'        => null,
		];
	}

	public function routerOut(DaoNavigationItem $navigation, &$urlParams): ?string
	{
		if (isset($urlParams['id'])) {
			$endSuffix = NavigationConfig::load('urlSuffix');
			$article   = $this->articlesService->getForNavigation(
				(int) $urlParams['id'],
				$urlParams['locale'] ?: $navigation->getLang(),
			);

			if (!in_array($article['category'], $navigation->componentParams['categories'])) {
				return null;
			}

			if ($article) {
				$articleUrl       = $article['alias'];
				$addCategoryAlias = isset($navigation->componentParams['addCategoryAlias']) && $navigation->componentParams['addCategoryAlias'] == 1;

				if (!BlogConfig::load('removeArticleIdFromUrl', false)) {
					$articleUrl = $article['id'] . '-' . $articleUrl;
				}

				return '/' . $navigation->alias . ($addCategoryAlias ? '/' . $article['categoryAlias'] : '') . '/' . $articleUrl . ($endSuffix ?: '');
			}
		}

		$urlParts = [];
		if (!$navigation->isHomepage) {
			$urlParts[] = $navigation->alias;
		}

		return '/' . implode('/', $urlParts);
	}

	/**
	 * @inheritdoc
	 */
	public function getComponentChild(string $lang, $navigation = null, $siteMapData = false)
	{
		$arr = [];

		foreach ($this->articlesService->getAll($navigation->componentParams['categories'] ?? null) as $article) {
			$data = [
				'action'    => $this->actionArticleDetail,
				'id'        => $article->getId(),
				'presenter' => $this->presenterArticleDetail,
			];

			if ($siteMapData) {
				$siteMapUrl = new SiteMapUrl;
				if ($article->getModified()) {
					$siteMapUrl->setModified($article->getModified());
				}
				$siteMapUrl->setSeo($article->getSeo());

				if ($article->getGallery()) {
					foreach ($article->getGallery()->getImages() as $img) {
						$siteMapUrl->addImage($img);
					}
				}

				$data['siteMapData']['default'] = $siteMapUrl;
			}

			if ($navigation) {
				$data['activeNavigation'] = $navigation;
			}

			$arr[] = $data;
		}

		return $arr;
	}
}
