<?php declare(strict_types = 1);

namespace Blog\Model\Rss;

use DOMDocument;
use DOMElement;
use Exception;
use InvalidArgumentException;
use Nette;
use Nette\InvalidStateException;
use Nette\Utils\DateTime;
use Nette\Utils\Strings;
use SimpleXMLElement;
use Tracy\Debugger;

class Generator
{
	use Nette\SmartObject;

	/** @var callable|null */
	protected           $dataSource         = null;
	protected string    $title              = '';
	protected string    $description        = '';
	protected string    $link               = '';
	protected string    $dateTimeFormat     = 'Y-m-d H:i:s';
	protected int       $shortenDescription = 150;
	protected ?DateTime $lastBuildDate      = null;

	public function getTitle(): string
	{
		return $this->title;
	}

	public function setTitle(string $title): void
	{
		$this->title = $title;
	}

	public function getDescription(): string
	{
		return $this->description;
	}

	public function setDescription(string $description): void
	{
		$this->description = $description;
	}

	public function getLink(): string
	{
		return $this->link;
	}

	public function setLink(string $link): void
	{
		$this->link = $link;
	}

	public function setDataSource(callable $dataSource): void
	{
		$this->dataSource = $dataSource;
	}

	public function getShortenDescription(): int
	{
		return $this->shortenDescription;
	}

	public function setShortenDescription(int $value): void
	{
		$this->shortenDescription = $value;
	}

	public function getDateTimeFormat(): string
	{
		return $this->dateTimeFormat;
	}

	public function setDateTimeFormat(string $format): void
	{
		$this->dateTimeFormat = $format;
	}

	public function getLastBuildDate(): ?DateTime
	{
		return $this->lastBuildDate;
	}

	public function setLastBuildDate(DateTime $lastBuildDate): void
	{
		$this->lastBuildDate = $lastBuildDate;
	}

	/**
	 * @throws InvalidArgumentException
	 */
	public function generate(): SimpleXMLElement
	{
		$doc = new DOMDocument;
		if (is_null($this->dataSource)) {
			throw new InvalidStateException('Data source for RSS generator is not set.');
		}
		$items   = call_user_func($this->dataSource);
		$channel = simplexml_load_file(__DIR__ . '/template.xml');

		if (!$channel) {
			Debugger::log("Failed create RSS chanel");
			throw new Exception("Failed create RSS chanel");
		}

		$NS = [
			'szn' => 'https://www.seznam.cz',
		];
		$channel->registerXPathNamespace('szn', $NS['szn']);

		if ($this->link) {
			$channel->channel->link[0][0] = $this->link;
		}
		if ($this->getLastBuildDate()) {
			$channel->channel->lastBuildDate = $this->getLastBuildDate()->format($this->dateTimeFormat);
		}
		if ($this->title) {
			$channel->channel->title[0][0] = $this->title;
		}
		if ($this->description) {
			$channel->channel->description[0][0] = $this->description;
		}

		/** @var RssChannelItem $item */
		foreach ($items as $item) {
			/** @var SimpleXMLElement $i */
			$i = $channel->channel->addChild('item');
			$i->addChild('title', $item->getTitle());
			$i->addChild('link', $item->getLink());
			$i->addChild('guid', $item->getLink());
			$i->addChild('image', $item->getImage(), $NS['szn']);
			$i->addChild('pubDate', $item->getPubDate()->format($this->getDateTimeFormat()));

			$desc = $i->addChild('description');

			if ($desc) {
				/** @var DOMElement|null $node */
				$node = dom_import_simplexml($desc);

				if ($node) {
					$no = $node->ownerDocument;
					$node->appendChild(
						$no->createCDATASection(
							trim(Strings::truncate($item->getDescription(), $this->getShortenDescription())),
						),
					);
				}
			}
		}

		return $channel;
	}

	public function response(): ?RssResponse
	{
		return new RssResponse($this->generate());
	}

}
