<?php declare(strict_types = 1);

namespace Blog\AdminModule\Components\Article;

use Blog\AdminModule\Model\Categories;
use Blog\Model\ArticleHits;
use Blog\AdminModule\Model\Articles as AdminArticles;
use Blog\Model\Articles;
use Blog\Model\Entities\Article;
use Blog\Model\Entities\ArticleInCategory;
use Blog\Model\OpenedArticlesService;
use Core\AdminModule\Model\Sites;
use Core\Components\Flashes\Flashes;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Doctrine\Common\Collections\Criteria;
use Gallery\Model\Entities\Album;
use Gallery\Model\Entities\Image;
use Core\Model\Entities\QueryBuilder;
use Nette\Application\AbortException;
use Nette\Utils\Html;
use Ublaboo\DataGrid\DataSource\DoctrineDataSource;

class ArticlesGrid extends BaseControl
{
	/** @var Image[] */
	protected array $articleImages = [];
	private AdminArticles $articlesAdminService;
	protected Articles $articles;
	protected ArticleHits $articleHitsService;
	protected OpenedArticlesService $openedArticlesService;
	protected Sites $sitesService;
	protected Categories $categories;
	protected array $articleHits = [], $openedArticles = [];

	public function __construct(AdminArticles $articlesAdminService, Articles $articles, ArticleHits $articleHits, Categories $categories, OpenedArticlesService $openedArticlesService, Sites $sites)
	{
		$this->articlesAdminService  = $articlesAdminService;
		$this->articles              = $articles;
		$this->articleHitsService    = $articleHits;
		$this->openedArticlesService = $openedArticlesService;
		$this->sitesService          = $sites;
		$this->categories            = $categories;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		$grid->setTableClass('table-blog-articles');
		$grid->setTemplateFile(__DIR__ . '/ArticlesGrid_grid.latte');
		$grid->setStrictSessionFilterValues(false);

		$sites = $this->sitesService->getAll();

		$qb = $this->em->getRepository(Article::class)->createQueryBuilder('a')
			->addSelect('c, aic, ct')
			->leftJoin('a.texts', 'at')
			->leftJoin('a.categories', 'aic')
			->leftJoin('aic.category', 'c')
			->leftJoin('c.texts', 'ct');

		$grid->setDataSource($qb);
		$grid->setDefaultSort(['id' => 'DESC']);

		/** @var DoctrineDataSource $dataSource */
		$dataSource = $grid->getDataSource();
		$dataSource->onDataLoaded[] = function($items) {
			/** @var Article[] $items */
			$ids      = [];
			$albumIds = [];
			$hits     = [];
			$images   = [];
			$opened   = [];
			foreach ($items as $item) {
				$ids[] = $item->getId();
				if ($item->getGallery()) {
					$albumIds[] = $item->getGallery()->getId();
				}
			}

			foreach ($this->em->getRepository(Image::class)->createQueryBuilder('i')
				         ->join(Album::class, 'a', 'WITH', 'a.id = i.album')
				         ->where('i.album IN (:albums)')->setParameter('albums', $albumIds)
				         ->orderBy('i.position', 'ASC')
				         ->groupBy('a.id')
				         ->getQuery()->getResult() as $row) {
				/** @var Image $row */
				$albumId = $row->getAlbum()->getId();
				if (!isset($images[$albumId])) {
					$images[$albumId] = $row;
				}
				else if ($row->isCover) {
					$images[$albumId] = $row;
				}
			}

			foreach ($this->openedArticlesService->checkArticles($ids) as $v) {
				$opened[$v->article->getId()] = $v;
			}

			$this->articleHits    = $hits;
			$this->articleImages  = $images;
			$this->openedArticles = $opened;
		};

		// Columns
		$grid->addColumnText('image', 'blog.articlesGrid.image', '')->setRenderer(function(Article $row) {
			$image = $row->getGallery() ? $this->articleImages[$row->getGallery()->getId()] : null;
			if (!$image) {
				return '';
			}

			return Html::el('')
				->addHtml(Html::el('img', [
					'onMouseOver' => "showPicture('spt-{$row->getId()}', 1)",
					'onMouseOut'  => "showPicture('spt-{$row->getId()}', 0)",
					'src'         => $this->imagePipe->request($image->getFilePath(), '50x40'),
				]))
				->addHtml(Html::el('img', [
					'id'    => 'spt-' . $row->getId(),
					'class' => 'show-picture-target',
					'src'   => $this->imagePipe->request($image->getFilePath(), '400x400'),
				]));
		})->setAlign('center');
		$grid->addColumnText('title', 'blog.articlesGrid.title')
			->setRenderer([$this, 'gridRenderTitle']);
		$grid->addColumnText('hits', 'blog.articlesGrid.hits')
			->setRenderer([$this, 'gridRenderHits'])
			->setAlign('right');
		$grid->addColumnText('categories', 'blog.articlesGrid.category')->setRenderer(function(Article $row) {
			return implode(', ', array_map(static fn(ArticleInCategory $v) => $v->getCategory()->getText()->getTitle(), $row->categories->toArray()));
		});
		$grid->addColumnStatus('featured', 'blog.articlesGrid.featured')
			->setAlign('center')
			->setTemplate(__DIR__ . '/statusColumn.latte')
			->addOption(1, 'blog.article.recommend.recommend')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(120, 'blog.article.recommend.v120')->setIcon('clock')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(300, 'blog.article.recommend.v300')->setIcon('clock')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(600, 'blog.article.recommend.v600')->setIcon('clock')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(1440, 'blog.article.recommend.v1440')->setIcon('clock')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(2880, 'blog.article.recommend.v2880')->setIcon('clock')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(4320, 'blog.article.recommend.v4320')->setIcon('clock')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(5760, 'blog.article.recommend.v5760')->setIcon('clock')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'blog.article.cancelRecommendations')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridFeaturedChange'];
		$grid->addColumnText('state', 'blog.articlesGrid.state', 'isPublished')->setRenderer([$this,
			'gridRenderState']);
		$grid->addColumnDateTime('publishUp', 'blog.articlesGrid.publishUp')->setFormat('j. n. Y H:i')->setSortable();
		$grid->addColumnText('id', 'blog.articlesGrid.id')->setAlign('right')->setSortable();

		// Filters
		$grid->getColumn('title')->setFilterText()->setCondition(function(QueryBuilder $qb, $value) {
			$qb->andWhere('at.title LIKE :title')->setParameter('title', "%$value%");
		});

		if (count($sites) > 1) {
			$flat = [];
			foreach ($sites as $site) {
				$tmp = $this->categories->getFlatTree($site->getIdent());

				if (empty($tmp)) {
					continue;
				}

				$flat[] = [
					'id'     => $tmp[0]['parent'],
					'parent' => 0,
					'name'   => $site->getIdent(),
				];

				$flat = array_merge($flat, $tmp);
			};
		} else {
			$flat = $this->categories->getFlatTree();
		}

		$grid->addFilterCheckboxNestedList('categoriesId', 'eshopCatalog.productGrid.defaultCategory', $flat)
			->setCondition(function(QueryBuilder $qb, $value) {
				$qb->andWhere('aic.category IN (:categoriesId)')
					->setParameter('categoriesId', (array) $value);
			});
		$grid->getColumn('categories')->setFilterText()->setCondition([$this, 'conditionGridCategories']);;

		// Actions
		$grid->addAction('delete', '', 'delete')->setIcon('times')->addClass('ajax')->setBsType('danger')->setConfirm('blog.articlesGrid.reallyDelete');
		$grid->addGroupAction('default.deleteSelected')->onSelect[] = [$this, 'handleDelete'];

		// Columns prototype
		$grid->getColumn('id')->getElementPrototype('th')->class[]       = 'w1nw';
		$grid->getColumn('image')->getElementPrototype('th')->class[]    = 'w1nw';
		$grid->getColumn('featured')->getElementPrototype('th')->class[] = 'w1nw';

		return $grid;
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	/**
	 * @param int[]|string[]|int|string $id
	 * @throws AbortException
	 */
	public function handleDelete($id): void
	{
		$presenter = $this->presenter;

		$result = is_array($id) ? $this->articlesAdminService->removeArticles($id) : $this->articlesAdminService->removeArticle($id);

		if (is_bool($result)) {
			if ($result) {
				$presenter->flashMessageSuccess('blog.article.removed');
			} else {
				$presenter->flashMessageDanger('blog.article.removeFailed');
			}
		}
		else
			if (count($result) === 0) {
				$presenter->flashMessageSuccess('blog.article.selectedRemoved');
			}
			else {
				$presenter->flashMessageInfo($this->t('blog.article.removedX', null, ['number' => count($result)]));
			}


		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	public function handleInvertPublish(int $id, string $lang): void
	{
		$presenter = $this->presenter;

		if ($this->articlesAdminService->invertPublish($id, $lang)) {
			$presenter->flashMessageSuccess('default.publishChanged');
		} else {
			$presenter->flashMessageDanger('default.publishChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function conditionGridCategories(QueryBuilder $qb, string $value): void
	{
		$criteria = Criteria::create();

		if ($value === '-') {
			$criteria->orWhere(Criteria::expr()->isNull('ct'));
		} else {
			foreach (explode(',', $value) as $val) {
				$criteria->orWhere(Criteria::expr()->contains('ct.title', trim($val)));
			}
		}

		$qb->addCriteria($criteria);
	}

	public function gridRenderTitle(Article $row): void
	{
		$opened = $this->openedArticles[$row->getId()] ?? null;
		if ($opened && $opened->user->getId() != $this->presenter->getUser()->getId()) {
			echo Html::el('i', ['class'               => 'fa fa-lock mr-2', 'data-toggle' => 'tooltip',
			                    'data-original-title' => $this->t('blog.article.isEditingBy', null, ['user' => $opened->user->name]) . ' - ' . $opened->lastActivity->format('H:i:s')]);
			echo Html::el('span')->setText($row->getText()->getTitle());
		} else {
			echo Html::el('a', ['href' => $this->presenter->link('Articles:edit', [$row->getId()])])
					 ->setHtml(!$row->getText() ? '<i>' . $this->t('blog.articlesGrid.titleNotFilled', null, ['lang' => $this->translator->getLocale()]) . '</i>' : $row->getText()
																																										->getTitle());
		}
	}

	public function gridRenderHits(Article $row): string
	{
		return number_format($row->shows ?? 0, 0, '', ' ');
	}

	public function gridRenderState(Article $row): Html
	{
		$wrap = Html::el('div class=grid-langs');

		foreach ($this->langsService->getLangs(false) as $lang) {
			$text = $row->getText($lang->getTag());
			$el   = Html::el('div class=grid-langs__lang');
			if ($text) {
				$a = Html::el('a', [
					'class' => 'ajax btn btn-xs ' . ($text->isPublished ? 'btn-success' : 'btn-danger'),
					'href'  => $this->link('invertPublish!', [$row->getId(), $lang->getTag()]),
				])->setText($lang->getShortTitle());
				$el->addHtml($a);
			}
			$wrap->addHtml($el);
		}

		return $wrap;
	}

	/**
	 * @param int $id
	 * @param int $newStatus
	 * @throws AbortException
	 */
	public function gridFeaturedChange($id, $newStatus): void
	{
		$presenter = $this->presenter;

		if ($this->articles->setFeature($id, $newStatus)) {
			$presenter->flashMessage('blog.article.featuredStatusChanged', Flashes::FLASH_SUCCESS);
		} else {
			$presenter->flashMessage('blog.article.featuredStatusChangeFailed', Flashes::FLASH_SUCCESS);
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

}
