<?php declare(strict_types = 1);

namespace Blog\Model\Entities;

use Core\Model\Entities\TTranslateListener;
use Core\Model\Helpers\Traits\TExtraField;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Gallery\Model\Entities\Album;
use Gallery\Model\Entities\Image;
use Core\Model\Entities\TId;
use Nette\Utils\DateTime;
use Tags\Model\Entities\Tag;

/**
 * @ORM\Table(name="blog__article")
 * @ORM\Entity
 * @ORM\EntityListeners({"ArticleListener", "Core\Model\Entities\TranslateListener"})
 */
class Article
{
	use TId;
	use TTranslateListener;
	use TExtraField;

	public const EXTRA_FIELD_SECTION = 'blogArticle';

	/**
	 * @var Collection<ArticleInCategory>
	 * @ORM\OneToMany(targetEntity="ArticleInCategory", mappedBy="article", indexBy="category")
	 */
	public Collection $categories;

	/**
	 * @var Collection<Tag>
	 * @ORM\ManyToMany(targetEntity="Tags\Model\Entities\Tag", fetch="EXTRA_LAZY")
	 * @ORM\JoinTable(name="blog__article_tags",
	 *     joinColumns={@ORM\JoinColumn(name="article_id", referencedColumnName="id", onDelete="CASCADE")},
	 *     inverseJoinColumns={@ORM\JoinColumn(name="tag_id", referencedColumnName="id", onDelete="CASCADE")}
	 *     )
	 */
	protected Collection $tags;

	/**
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	protected ?string $image = null;

	/**
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	public ?string $video = null;

	/**
	 * @ORM\OneToOne(targetEntity="Gallery\Model\Entities\Album")
	 * @ORM\JoinColumn(name="gallery_id", referencedColumnName="id", onDelete="SET NULL")
	 */
	public ?Album $gallery = null;

	/**
	 * @var \DateTimeInterface
	 * @ORM\Column(type="datetime", nullable=true)
	 */
	public $publishUp;

	/**
	 * @var DateTime
	 * @ORM\Column(type="datetime", nullable=true)
	 */
	public $publishDown;

	/**
	 * @ORM\Column(type="string", length=60, nullable=true)
	 */
	private ?string $featured = null;

	/**
	 * @ORM\Column(type="array", nullable=true)
	 */
	public ?array $params = null;

	/**
	 * @ORM\Column(type="integer", options={"unsigned"=true, "default"=0})
	 */
	public int $shows;

	/**
	 * @var Collection<string, ArticleText>
	 * @ORM\OneToMany(targetEntity="ArticleText", mappedBy="article", indexBy="lang")
	 */
	protected Collection $texts;

	/**
	 * @ORM\Column(type="string", length=60, nullable=true)
	 */
	public ?string $layout = null;

	public function __construct()
	{
		$this->texts       = new ArrayCollection;
		$this->shows       = 0;
		$this->categories  = new ArrayCollection;
		$this->extraFields = new ArrayCollection;
	}

	public function setTexts(array $texts): self
	{
		$this->texts = new ArrayCollection($texts);

		return $this;
	}

	public function getText(?string $lang = null): ?ArticleText { return $this->texts[$lang ?: $this->locale] ?? null; }

	/**
	 * @return Collection<string, ArticleText>
	 */
	public function getTexts(): Collection
	{
		return $this->texts;
	}

	public function setId(?int $id): void
	{
		$this->id = $id;
	}

	/******
	 * === Image
	 */

	public function getImage(): ?Image
	{
		return $this->gallery && $this->gallery->getCoverImage() ? $this->gallery->getCoverImage() : null;
	}

	public function getImageFile(): ?string
	{
		return $this->getImage() ? $this->getImage()->getFilePath() : null;
	}

	public function getGallery(): ?Album
	{
		return $this->gallery;
	}

	/*******
	 * === Featured
	 */

	public function getFeatured(): ?string
	{
		return $this->featured;
	}

	public function getIsFeatured(): bool
	{
		return $this->featured !== null;
	}

	public function setFeatured(?string $featured): self
	{
		$this->featured = $featured;

		return $this;
	}

	/******
	 * === Tags
	 */

	public function setTags(array $tags): void
	{
		$this->tags = new ArrayCollection($tags);
	}

	/**
	 * @return Collection<Tag>
	 */
	public function getTags(): Collection
	{
		return $this->tags;
	}

	public function getTagsId(): array
	{
		return $this->tags->map(fn(Tag $tag) => $tag->getId())->toArray();
	}
}
