<?php declare(strict_types = 1);

namespace Blog\Model\Entities;

use Core\Model\Entities\TTranslateListener;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Core\Model\Entities\TId;

/**
 * @Gedmo\Tree(type="nested")
 * @ORM\Table(name="blog__category")
 * @ORM\Entity(repositoryClass="Core\Model\Entities\Repository\NestedTreeRepository")
 * @ORM\EntityListeners({"CategoryListener", "Core\Model\Entities\TranslateListener"})
 */
class Category
{
	use TId;
	use TTranslateListener;

	/**
	 * @ORM\Column(type="string", length=60)
	 */
	public string $layout;

	/**
	 * @ORM\Column(type="string", length=60, nullable=true)
	 */
	public ?string $password = null;

	/**
	 * @var int
	 * @ORM\Column(type="smallint", options={"default": 1})
	 */
	public $isPublished;

	/**
	 * @var Collection<Category>
	 * @ORM\OneToMany(targetEntity="Category", mappedBy="parent")
	 * @ORM\OrderBy({"lft" = "ASC"})
	 */
	protected Collection $children;

	/**
	 * @Gedmo\TreeParent
	 * @ORM\ManyToOne(targetEntity="Category", inversedBy="childrens", cascade={"persist"})
	 * @ORM\JoinColumn(name="parent_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public ?Category $parent;

	/**
	 * @Gedmo\TreeLeft
	 * @ORM\Column(name="lft", type="integer")
	 */
	protected int $lft;

	/**
	 * @Gedmo\TreeLevel
	 * @ORM\Column(name="lvl", type="integer")
	 */
	protected int $lvl;

	/**
	 * @Gedmo\TreeRight
	 * @ORM\Column(name="rgt", type="integer")
	 */
	protected int $gt;

	/**
	 * @Gedmo\TreeRoot
	 * @ORM\ManyToOne(targetEntity="Category")
	 * @ORM\JoinColumn(name="tree_root", referencedColumnName="id", onDelete="CASCADE")
	 */
	protected ?Category $root;

	/**
	 * @var Collection<CategoryText>
	 * @ORM\OneToMany(targetEntity="CategoryText", mappedBy="category", indexBy="lang")
	 */
	protected Collection $texts;

	public function __construct()
	{
		$this->texts       = new ArrayCollection;
		$this->layout      = 'default';
		$this->children    = new ArrayCollection;
		$this->isPublished = 1;
	}

	public function getLvl(): int { return $this->lvl; }
	public function getRoot(): ?Category { return $this->root; }

	public function setPassword(?string $password): void
	{
		$this->password = $password ?: null;
	}

	public function isLocked(): bool { return $this->password ? true : false; }
	public function getLayout(): string { return $this->layout ?: 'default'; }

	public function setLayout(string $layout): self
	{
		$this->layout = $layout;

		return $this;
	}

	public function setTexts(array $texts): self
	{
		$this->texts = new ArrayCollection($texts);

		return $this;
	}

	public function getText(?string $lang = null): ?CategoryText { return $this->texts[$lang ?: $this->locale] ?? null; }

	/**
	 * @return Collection<CategoryText>
	 */
	public function getTexts() { return $this->texts; }
}
