<?php declare(strict_types = 1);

namespace Blog\Model\Entities;

use Blog\Model\Articles;
use Blog\Model\Navigation\Blog;
use Core\Model\Images\ImagePipe;
use Doctrine;
use Doctrine\ORM\Event\LifecycleEventArgs;
use Doctrine\ORM\Mapping as ORM;
use Kdyby\Events\Subscriber;
use Navigations\Model\Navigations;
use Nette\Application\LinkGenerator;
use Nette\Caching\Cache;
use Nette\Caching\IStorage;
use Nette\Http\Session;
use Nette\SmartObject;

class ArticleListener implements Subscriber
{
	use SmartObject;

	/** @var int */
	protected static $i = 0;

	/** @var Cache */
	protected $cache;

	/** @var Cache */
	protected $navigationCache;

	/** @var Session */
	protected $session;

	protected $fields = false;

	public function __construct(IStorage $cacheStorage, Session $session)
	{
		$this->session         = $session;
		$this->cache           = new Cache($cacheStorage, Articles::CACHE_NAMESPACE);
		$this->navigationCache = new Cache($cacheStorage, Navigations::CACHE_NAMESPACE);
	}

	public function getSubscribedEvents()
	{
		return [];
	}

	/**
	 * @ORM\PreRemove
	 *
	 * @param Article                               $article
	 * @param Doctrine\ORM\Event\LifecycleEventArgs $event
	 */
	public function onPreRemove($article, $event)
	{
		$sessionSection = $this->session->getSection('unsavedArticles');
		unset($sessionSection->{'article_' . $article->getId()});
	}

	/**
	 * @ORM\PostPersist
	 * @ORM\PostUpdate
	 * @ORM\PostRemove
	 *
	 * @param Article            $article
	 * @param LifecycleEventArgs $event
	 */
	public function postHandler(Article $article, LifecycleEventArgs $event)
	{
		foreach ($article->getTexts()->getKeys() as $l) {
			$this->cache->remove('article/' . $article->getId() . '/' . $l);
		}

		$this->navigationCache->clean([Cache::TAGS => 'article/' . $article->getId()]);

		$changeSet = $event->getEntityManager()->getUnitOfWork()->getEntityChangeSet($article);
		if (isset($changeSet['featured'])) {
			$this->cache->clean([Cache::TAGS => [Articles::CACHE_NAMESPACE . '/featured']]);
		}
	}
}
