<?php declare(strict_types = 1);

namespace Ceskaposta\Model\Subscribers;

use Ceskaposta\Model\Entities\PostOfficeOrder;
use Ceskaposta\Model\PostOffices;
use Contributte\EventDispatcher\EventSubscriber;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\FrontModule\Model\Event\OrderFormEvent;
use EshopOrders\FrontModule\Model\Event\SaveOrderFormDataEvent;
use EshopOrders\FrontModule\Model\Event\SetOrderFormDataEvent;
use EshopOrders\Model\Entities\Spedition;
use Kdyby\Doctrine\EntityManager;
use Nette\Http\Request;
use Nette\Localization\ITranslator;

class OrderFormSubscriber implements EventSubscriber
{
	/** @var array */
	protected $conf;

	/** @var EntityManager */
	protected $em;

	/** @var PostOffices */
	protected $postOfficeService;

	/** @var ITranslator */
	protected $translator;

	/** @var Request */
	protected $httpRequest;

	public function __construct(EntityManager $em, ITranslator $translator, PostOffices $postOffices, Request $request)
	{
		$this->em                = $em;
		$this->translator        = $translator;
		$this->postOfficeService = $postOffices;
		$this->httpRequest       = $request;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.createOrderForm'        => ['createOrderForm', 100],
			'eshopOrders.setOrderFormData'       => ['setOrderFormData', 100],
			'eshopOrders.orderFormValidate'      => ['orderFormValidate', 100],
			'eshopOrders.orderBeforeSave'        => ['orderBeforeSave', 100],
			'eshopOrders.saveOrderFormDataStep2' => ['saveOrderFormDataStep2', 100],
			'eshopOrders.saveOrderFormDataStep3' => ['saveOrderFormDataStep3', 100],
		];
	}

	public function createOrderForm(OrderFormEvent $event): void
	{
		$form = $event->form;

		$form->addText('postOfficeName', 'blog.articleForm.tags')
			->setPlaceholder('ceskapostaFront.selectCeskaPosta');
		$form->addHidden('postOfficePsc')->setHtmlAttribute('id', 'postOfficePsc')->getControlPrototype()->class[] = 'orderFormPostOfficePsc';
	}

	public function setOrderFormData(SetOrderFormDataEvent $event): void
	{
		$data    = $event->data;
		$form    = $event->form;
		$value   = $form->getValues();
		$request = $this->httpRequest;

		if ($form->getComponent('postOfficeName', false))
			$form['postOfficeName']->setDefaultValue($data['postOfficeName']);
		if ($form->getComponent('postOfficePsc', false))
			$form['postOfficePsc']->setDefaultValue($data['postOfficePsc']);

		$spedition = $request->getPost('spedition');
		if (!in_array($spedition, $this->getCeskaPostaBNPSpeditionIds()))
			return;

		$posOffice = $this->postOfficeService->getPostOffices($data['postOfficePsc']);
		if (!$posOffice[0])
			return;

		$posOffice = $posOffice[0];

		foreach (['company', 'firstName', 'lastName', 'email', 'phone'] as $k) {
			$form[$k . '2']->setValue($value[$k]);
		}

		$form['useAddrDeli']->setValue(true);
		$form['street2']->setValue($posOffice->getStreet());
		$form['city2']->setValue($posOffice->village);
		$form['postal2']->setValue($posOffice->psc);
		$form['country2']->setValue('CZ');
	}

	public function saveOrderFormDataStep2(SaveOrderFormDataEvent $event): void
	{
		$data    = &$event->data;
		$request = $this->httpRequest;

		$spedition = $request->getPost('spedition');

		if (in_array($spedition, $this->getCeskaPostaBNPSpeditionIds())) {
			foreach (['postOfficeName', 'postOfficePsc'] as $k) {
				$data[$k] = $request->getPost($k);
			}

			$data['disableDeliveryAddressSpedition'] = 'postOffice';
			$data['disableDeliveryAddress']          = true;
		} else if ($data['disableDeliveryAddressSpedition'] == 'postOffice') {
			unset($data['disableDeliveryAddress']);
		}
	}

	public function saveOrderFormDataStep3(SaveOrderFormDataEvent $event): void
	{
		$data    = &$event->data;
		$request = $this->httpRequest;

		$spedition = $request->getPost('spedition');
		if (!in_array($spedition, $this->getCeskaPostaBNPSpeditionIds()))
			return;

		$posOffice = $this->postOfficeService->getPostOffices($data['postOfficePsc']);
		if (!$posOffice[0])
			return;

		$posOffice = $posOffice[0];

		foreach (['company', 'firstName', 'lastName', 'email', 'phone'] as $k)
			$data[$k . '2'] = $data[$k];

		$data['useAddrDeli'] = true;
		$data['street2']     = $posOffice->getStreet();
		$data['city2']       = $posOffice->village;
		$data['postal2']     = $posOffice->psc;
		$data['country2']    = 'CZ';
	}

	public function orderFormValidate(OrderFormEvent $event): void
	{
		$form         = $event->form;
		$values       = $form->getValues();
		$speditionIds = $this->getCeskaPostaBNPSpeditionIds();

		if (in_array($values['spedition'], $speditionIds) && !$values['postOfficePsc'])
			$form['postOfficeName']->addError($this->translator->translate('ceskaposta.orderForm.postOfficePscMissing'));
	}

	public function orderBeforeSave(OrderEvent $event): void
	{
		$values = $event->formData;

		if (in_array($values['spedition'], $this->getCeskaPostaBNPSpeditionIds())) {
			$entity    = new PostOfficeOrder($event->order, $values['postOfficePsc']);
			$posOffice = $this->postOfficeService->getPostOffices($values['postOfficePsc']);
			if (count($posOffice) == 1) {
				$entity->postName    = $posOffice[0]->namePostOffice;
				$entity->village     = $posOffice[0]->village;
				$entity->partVillage = $posOffice[0]->partVillage;
				$this->em->persist($entity);
			}
		}
	}

	protected function getCeskaPostaBNPSpeditionIds(): array
	{
		$result = [];
		foreach ($this->em->getRepository(Spedition::class)->createQueryBuilder('s')->select('s.id, s.ident')
			         ->where('s.ident = :ident')->setParameter('ident', 'cpbnp')
			         ->getQuery()->useResultCache(true, 30)->getArrayResult() as $row)
			$result[$row['ident']] = $row['id'];

		return $result;
	}
}
