<?php declare(strict_types = 1);

namespace Ceskaposta\AdminModule\Model\Subscribers;

use Ceskaposta\Model\CeskaPostaConfig;
use Ceskaposta\Model\Entities\ICeskaPostaOrder;
use Ceskaposta\Model\Entities\ParcelDeliveryToHandOrder;
use Ceskaposta\Model\Entities\PostOfficeOrder;
use Ceskaposta\Model\Entities\PostWarehouseOrder;
use Ceskaposta\Model\Helper;
use Ceskaposta\Model\OrderExpeditionHelper;
use Ceskaposta\Model\OrdersExported;
use Ceskaposta\Model\ParcelsService;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\ControlEvent;
use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\Event;
use Core\Model\Event\EventDispatcher;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\GridFilterEvent;
use Core\Model\Event\SetFormDataEvent;
use Core\Model\UI\Form\BaseForm;
use Doctrine\Common\Collections\Criteria;
use EshopOrders\AdminModule\Components\Order\OrderForm;
use EshopOrders\AdminModule\Components\Order\OrdersGrid;
use EshopOrders\AdminModule\Components\Order\OrderSpeditionForm;
use EshopOrders\AdminModule\Components\Order\OrderStatusesGrid;
use EshopOrders\AdminModule\Model\Dao\OrderExpedition;
use EshopOrders\AdminModule\Model\Event\OrderStatusEvent;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderSpedition;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Entities\Spedition;
use EshopOrders\Model\EshopOrdersConfig;
use Exception;
use Nette\Utils\Html;
use Nette\Utils\Strings;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Tracy\Debugger;
use EshopOrders\AdminModule\Components\Expedition;

class OrderSubscriber implements EventSubscriberInterface
{
	/** @var ICeskaPostaOrder[]|null */
	protected static ?array $allOrders = null;
	protected OrdersExported         $ordersExported;
	protected ParcelsService         $parcelsService;
	protected EntityManagerDecorator $em;
	protected EventDispatcher        $eventDispatcher;

	public function __construct(
		OrdersExported         $ordersExported,
		ParcelsService         $parcelsService,
		EntityManagerDecorator $em,
		EventDispatcher        $eventDispatcher
	)
	{
		$this->ordersExported  = $ordersExported;
		$this->parcelsService  = $parcelsService;
		$this->em              = $em;
		$this->eventDispatcher = $eventDispatcher;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			OrderForm::class . '::onAttach'                                        => 'orderFormAttached',
			OrdersGrid::class . '::columnSpeditionRender'                          => 'columnSpeditionRender',
			OrdersGrid::class . '::filterPackageNumber'                            => 'gridFilterPackageNumber',
			Expedition\OrdersGrid::class . '::createForm-cpost'                    => 'expeditionCreateForm',
			Expedition\ExpeditionAdvancedOptionsForm::class . '::createForm-cpost' => 'expeditionCreateForm',
			OrderSpeditionForm::class . '::createForm'                             => 'orderSpeditionCreateForm',
			OrderSpeditionForm::class . '::formSuccess'                            => 'orderSpeditionFormSuccess',
			OrderSpeditionForm::class . '::setOrder'                               => 'orderSpeditionSetOrder',
			OrderStatusesGrid::class . '::beforeDelete'                            => 'orderStatusDelete',
		];
	}

	public function orderFormAttached(ControlEvent $event): void
	{
		/** @var OrderForm $control */
		$control   = $event->control;
		$order     = $control->order;
		$spedition = $order->getSpedition() ? $order->getSpedition()->getSpedition() : null;

		if ($spedition && $spedition->getIdent() === 'cpost') {
			$control->template->ceskapostaExport            = $this->ordersExported->getOrdersExported([$order->getId()])[$order->getId()] ?? null;
			$control->template->speditionExportInfoTemplate = __DIR__ . '/exportInfoTemplate.latte';
		}
	}

	public function columnSpeditionRender(Event $event): void
	{
		$data           = $event->data;
		$speditionIdent = 'cpost';
		$export         = null;

		/** @var Order $order */
		$order = $data['order'];

		/** @var Html $html */
		$html = $data['html'];

		if ($data['ordersBySpedition'][$speditionIdent] ?? null) {
			if (self::$allOrders === null) {
				self::$allOrders = $this->ordersExported->getOrdersExported(array_keys($data['ordersBySpedition'][$speditionIdent] ?? []), false);
			}

			$export = self::$allOrders[$order->getId()] ?? null;
		} else if ($order->getSpeditionIdent() === $speditionIdent) {
			$export = $this->ordersExported->getOrdersExported([$order->getId()]);
		}

		if ($export) {
			$h     = $html->addHtml(Html::el('div'));
			$urls  = $export->getTrackingUrls();
			$count = count($urls);
			$i     = 1;
			foreach ($urls as $number => $url) {
				$h->addHtml(Html::el('a', ['target' => '_blank'])
					->setAttribute('href', $url)
					->setText($number)
				);
				if ($i !== $count) {
					$h->addText(', ');
				}
				$i++;
			}
		}
	}

	public function gridFilterPackageNumber(GridFilterEvent $event): void
	{
		if (!CeskaPostaConfig::load('enable')) {
			return;
		}

		$ids = $this->ordersExported->findIdByPackageNumber($event->value);

		if ($ids) {
			$event->criteria->orWhere(Criteria::expr()->in('o.id', $ids));
		}
	}

	public function orderSpeditionCreateForm(CreateFormEvent $event): void
	{
		/** @var OrderSpeditionForm $control */
		$control = $event->control;
		$form    = $event->form;

		$cpIds = [];
		foreach ($control->getSpeditions() as $sped) {
			if ($sped->getIdent() === 'cpost') {
				$cpIds[$sped->isPickup ? 'isPickup' : 'notPickup'][] = $sped->getId();
			}
		}

		if (empty($cpIds)) {
			return;
		}

		$container = $form->addContainer('ceskaposta', '');
		$container->addText('parcelId', 'ceskaposta.entity.parcelId')
			->setDescription('ceskaposta.entity.parcelsList');
		$container->addText('numberPackage', 'ceskaposta.entity.numberPackage')
			->setDescription('ceskaposta.entity.numberPackageDesc');
		$container->addBool('removeExport', 'ceskaposta.entity.removeExportStatus')
			->setDefaultValue(0);
		$container->addBool('updateDeliveryAddress', 'ceskaposta.entity.updateDeliveryAddress')
			->setDefaultValue(1);

		$form->getComponent('spedition')->addCondition($form::IS_IN, array_merge($cpIds['isPickup'], $cpIds['notPickup']))
			->toggle($form['ceskaposta']['numberPackage']->getHtmlId())
			->toggle($form['ceskaposta']['removeExport']->getHtmlId())
			->toggle($form['ceskaposta']['updateDeliveryAddress']->getHtmlId());
		$form->getComponent('spedition')->addCondition($form::IS_IN, $cpIds['isPickup'])
			->toggle($form['ceskaposta']['parcelId']->getHtmlId());
		$container->getComponent('parcelId')->addConditionOn($form['spedition'], $form::IS_IN, $cpIds['isPickup'])
			->setRequired();
	}

	public function orderSpeditionFormSuccess(FormSuccessEvent $event): void
	{
		/** @var OrderSpedition $orderSpedition */
		$orderSpedition = $event->custom['entity'];
		/** @var OrderSpeditionForm $control */
		$control = $event->control;
		$orderId = $orderSpedition->getOrder()->getId();
		$values  = $event->values;
		/** @var Spedition|null $spedition */
		$spedition = null;
		/** @var ICeskaPostaOrder|null $cpOrder */
		$cpOrder = $this->ordersExported->getOrders(null, [$orderId])[$orderId] ?? null;

		$isCp = false;

		foreach ($control->getSpeditions() as $sped) {
			if ($sped->getIdent() === 'cpost' && $sped->getId() == $values->spedition) {
				$isCp      = true;
				$spedition = $sped;
				break;
			}
		}

		try {
			if ($isCp) {

				$onlyParcelIdChanged = $cpOrder && $cpOrder->getIdent() === $spedition->code1 && $cpOrder->getPostPSC() !== $values->ceskaposta->parcelId && !empty($cpOrder->getPostPSC()) && !empty($values->ceskaposta->parcelId);

				// pri zmene na cp, nebo z cp na cp (ale napr z balikovny na postu ci do ruky apod), z cp na cp (ale napr z balikovny X na balikovnu Y)
				if (!$cpOrder || $cpOrder->getIdent() !== $spedition->code1 || $onlyParcelIdChanged) {
					if ($spedition->isPickup) {
						$parcel = $this->parcelsService->getParcelByService($values->ceskaposta->parcelId, $spedition->code1);
						if ($parcel === null) {
							$event->form->addError('ceskaposta.orderForm.postOfficeNotFound');
							$control->redrawControl('form');

							return;
						}

						if ($cpOrder !== null && !$onlyParcelIdChanged) {
							$this->em->remove($cpOrder);
						}

						if ($spedition->code1 === 'napostu') {
							/** @var PostOfficeOrder $cpOrder */
							$cpOrder           = $onlyParcelIdChanged ? $cpOrder : new PostOfficeOrder($orderSpedition->getOrder(), $values->ceskaposta->parcelId);
							$cpOrder->postName = $parcel->naz_prov;
						} else if ($spedition->code1 === 'balikovna') {
							/** @var PostWarehouseOrder $cpOrder */
							$cpOrder           = $onlyParcelIdChanged ? $cpOrder : new PostWarehouseOrder($orderSpedition->getOrder(), $values->ceskaposta->parcelId);
							$cpOrder->postName = $parcel->nazev;
						} else {
							return;
						}

						$cpOrder->village     = $parcel->obec;
						$cpOrder->partVillage = $parcel->c_obce;
						$cpOrder->postPSC     = $values->ceskaposta->parcelId;

						$deliveryAddress = $cpOrder->getOrder()->getAddressDelivery();

						if ($deliveryAddress && $values['ceskaposta']->updateDeliveryAddress) {
							$deliveryAddress->setStreet($cpOrder->partVillage);
							$deliveryAddress->setPostal($cpOrder->postPSC);
							$deliveryAddress->setCity($cpOrder->village);
							$this->em->persist($deliveryAddress);
							if ($event->presenter && isset($event->presenter['orderForm'])) {
								$event->presenter['orderForm']->redrawControl('addressDelivery');
							}
						}
					} else {
						if ($cpOrder !== null) {
							$this->em->remove($cpOrder);
						}

						$cpOrder = new ParcelDeliveryToHandOrder($orderSpedition->getOrder());

						if ($event->presenter && $values['ceskaposta']->updateDeliveryAddress) {
							$event->presenter->flashMessageWarning('ceskaposta.orderForm.updateDeliveryAddress');
							$event->presenter->redrawControl('flashes');
						}
					}
				}

				$numberPackages      = trim(str_replace(' ', '', (string) $values->ceskaposta->numberPackage)) ?: null;
				$numberPackages      = $numberPackages ? trim($numberPackages, ',') : null;
				$numberPackages      = $numberPackages && Strings::contains($numberPackages, ',') ? array_map('trim', explode(',', $numberPackages)) : [];
				$numberPackagesCount = count($numberPackages);

				/** @var PostWarehouseOrder|ParcelDeliveryToHandOrder $cpOrder */
				if ($numberPackagesCount <= 1) {
					$cpOrder->numberPackage = $numberPackages[0] ?? null;
					$cpOrder->associatedNumberPackages->clear();
				} else {
					$existsPackages            = $cpOrder->associatedNumberPackages->getKeys();
					$existsPackagesWithoutMain = array_slice($numberPackages, 1, count($numberPackages) - 1);
					$diffToAdd                 = array_diff($existsPackagesWithoutMain, $existsPackages);
					$diffToRemove              = array_diff($existsPackages, $existsPackagesWithoutMain);

					for ($i = 0; $i < $numberPackagesCount; $i++) {
						$numberPackage = $numberPackages[$i];
						if ($i === 0) {
							$cpOrder->numberPackage = $numberPackage;
						} else {
							$class = Helper::getAssociatedClassByMainClass(get_class($cpOrder));
							if (in_array($numberPackage, $diffToAdd)) {
								$cpOrder->associatedNumberPackages->add(new $class($numberPackage, $cpOrder));
							}
						}
					}

					foreach ($diffToRemove as $p) {
						$cpOrder->associatedNumberPackages->remove($p);
					}
				}

				if ($cpOrder->getNumberPackage() && $cpOrder->getExported() === null) {
					$cpOrder->export();
				}

				if ($values->ceskaposta->removeExport === 1) {
					$cpOrder->resetExport();
				}

				$this->em->persist($cpOrder);

				$this->eventDispatcher->dispatch(new Event(['order'          => $cpOrder->getOrder(),
				                                            'numberPackages' => $numberPackages]), 'orderSubscriber.orderSpeditionFormSuccess');
			} else if ($cpOrder) {
				$this->em->remove($cpOrder);
			}
		} catch (Exception $e) {
			Debugger::log($e, 'orderSpeditionForm');
			$event->presenter->flashMessageDanger('ceskaposta.orderForm.speditionSaveError');
		}
	}

	public function orderSpeditionSetOrder(SetFormDataEvent $event): void
	{
		/** @var OrderSpedition $orderSpedition */
		$orderSpedition = $event->entity;
		$orderId        = $orderSpedition->getOrder()->getId();

		$cpOrder = $this->ordersExported->getOrders(null, [$orderId])[$orderId] ?? null;
		if (!$cpOrder) {
			return;
		}

		$event->form->getComponent('ceskaposta')->setDefaults([
			'parcelId'      => $cpOrder->getPostPSC(),
			'numberPackage' => implode(', ', $cpOrder->getAllNumberPackages()),
		]);
	}

	public function orderStatusDelete(OrderStatusEvent $event): void
	{
		if (!CeskaPostaConfig::load('enable')) {
			return;
		}

		$orderStatus = $event->orderStatus;
		$order       = $orderStatus->getOrder();

		if ($orderStatus->getStatus()->getId() !== OrderStatus::STATUS_SPEDITION || ($order->getSpeditionIdent() && $order->getSpeditionIdent() !== 'cpost')) {
			return;
		}

		$orderId = $order->getId();
		$cpOrder = $this->ordersExported->getOrders(null, [$orderId])[$orderId] ?? null;

		if (!$cpOrder) {
			return;
		}

		$cpOrder->resetExport();
		$this->em->persist($cpOrder);
		$this->em->flush($cpOrder);
	}

	public function expeditionCreateForm(CreateFormEvent $event): void
	{
		$form = $event->form;

		/** @var OrderExpedition|null $orderExpedition */
		$orderExpedition = $form->getCustomData('order');

		if ($orderExpedition && $orderExpedition->expeditionEntity && $orderExpedition->expeditionEntity instanceof PostWarehouseOrder) {
			$form->addCustomData('isHiddenQuantity', 1);
		}

		if (EshopOrdersConfig::load('expeditionOrdersGrid.enableAdvancedOptions')) {
			$form->setShowLangSwitcher(false);
			$form->addCustomData('advancedOptionsGridTemplate', __DIR__ . '/advancedOptionsGridTemplate.latte');

			$form->getComponent('quantity')->addCondition($form::FILLED)
				->addRule($form::MAX, null, CeskaPostaConfig::load('maxPackages'));

			$container = $form->addContainer('advancedOptions');
			$container->addSelect('packageSize', 'ceskaposta.expeditionForm.packageSize', OrderExpeditionHelper::getPackageSizes());
			$container->addText('weight', 'ceskaposta.expeditionForm.weight')
				->setDefaultValue($orderExpedition && $orderExpedition->order->getItemsWeight(true) ? $orderExpedition->order->getItemsWeight(true) : CeskaPostaConfig::load('weight'))
				->addCondition($form::FILLED)
				->addRule($form::FLOAT);
			if ($orderExpedition) {
				$container->addText('insuredValue', 'ceskaposta.expeditionForm.insuredValue')
					->setDefaultValue($orderExpedition->order->getPrice())
					->addCondition($form::FILLED)
					->addRule($form::FLOAT);
			}
			$container->addSelect('variableSymbol', 'ceskaposta.expeditionForm.variableSymbol.caption', OrderExpeditionHelper::getVariableSymbols());
			$container->addTextArea('exportNote', 'ceskaposta.expeditionForm.exportNote');
			$container->addCheckboxList('services', 'ceskaposta.expeditionForm.services', OrderExpeditionHelper::getServices());
		}

	}

}
