<?php declare(strict_types = 1);

namespace Ceskaposta\Model\Entities;

use Ceskaposta\Model\CeskaPostaConfig;
use Ceskaposta\Model\Entities\Attrs\IdTransaction;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use EshopOrders\Model\Entities\Order;
use Doctrine\ORM\Mapping as ORM;
use DateTime;

/**
 * @ORM\Table(name="ceskaposta__post_warehouse_order")
 * @ORM\Entity
 * @ORM\EntityListeners({"EshopOrders\Model\Listeners\CarrierListener"})
 */
class PostWarehouseOrder implements ICeskaPostaOrder
{
	public const PREFIX_PARCEL_CODE = 'NB';

	use IdTransaction;

	/**
	 * @ORM\Id
	 * @ORM\OneToOne(targetEntity="\EshopOrders\Model\Entities\Order")
	 * @ORM\JoinColumn(name="order_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	protected Order $order;

	/**
	 * @ORM\Column(name="post_psc", type="string")
	 */
	public string $postPSC;

	/**
	 * @ORM\Column(type="string")
	 */
	public string $postName;

	/**
	 * @ORM\Column(type="string")
	 */
	public string $village;

	/**
	 * @ORM\Column(type="string")
	 */
	public string $partVillage;

	/**
	 * @var DateTime|null
	 * @ORM\Column(type="datetime", nullable=true)
	 */
	private $exported;

	/**
	 * @ORM\Column(type="string", nullable=true)
	 */
	public ?string $numberPackage = null;

	/**
	 * @var Collection<string, ParcelNumberPostWarehouse>
	 * @ORM\OneToMany(targetEntity="ParcelNumberPostWarehouse", mappedBy="parcelOrder", cascade={"all"}, orphanRemoval=true, indexBy="numberPackage")
	 */
	public Collection $associatedNumberPackages;

	public function __construct(Order $order, string $postPSC)
	{
		$this->order    = $order;
		$this->postPSC = $postPSC;
		$this->associatedNumberPackages = new ArrayCollection;
	}

	public function getOrder(): Order { return $this->order; }

	public function getPostPSC(): string { return $this->postPSC; }

	public function export(?DateTime $datetime = null): self
	{
		$this->exported = $datetime ?? new DateTime;

		return $this;
	}

	public function getExported(): ?DateTime { return $this->exported; }

	public function getPostName(): ?string
	{
		return $this->postName;
	}

	public function getVillage(): ?string
	{
		return $this->village;
	}

	public function getPartVillage(): ?string
	{
		return $this->partVillage;
	}

	public function getNumberPackage(): ?string
	{
		return $this->numberPackage;
	}

	public function isExported(): bool
	{
		return $this->getExported() !== null;
	}

	public function isPickUp(): bool
	{
		return true;
	}

	public function getPrefixParcelCode(): string
	{
		return static::PREFIX_PARCEL_CODE;
	}

	public function getTrackingUrl(): string
	{
		return $this->numberPackage ? sprintf(CeskaPostaConfig::load('trackingUrl', ''), $this->numberPackage) : '';
	}

	public function getTrackingUrls(): array
	{
		$result = [];
		if (empty(!$this->getTrackingUrl())) {
			$result[$this->numberPackage] = $this->getTrackingUrl();
		}
		foreach ($this->getAssociatedNumberPackages() as $number) {
			$result[$number->numberPackage] = sprintf(CeskaPostaConfig::load('trackingUrl', ''), $number->numberPackage);
		}

		return $result;
	}

	public function resetExport(): void
	{
		$this->numberPackage = null;
		$this->exported      = null;
		$this->idTransaction = null;
		$this->associatedNumberPackages->clear();
	}

	public function getIdent(): string
	{
		return 'balikovna';
	}

	/**
	 * @return ParcelNumberPostWarehouse[]
	 */
	public function getAssociatedNumberPackages(): array
	{
		return $this->associatedNumberPackages->toArray();
	}

	public function getAllNumberPackages(): array
	{
		if (!$this->numberPackage) {
			return [];
		}
		return array_merge([$this->numberPackage], array_map(static fn(ICeskaPostaNumberPackage $parcel) => $parcel->getNumberPackage(), $this->getAssociatedNumberPackages()));
	}
}
