<?php

namespace Ceskaposta\Model\Libs;

use \Exception;

/**
* ČESKÁ POŠTA (B2B API)
*
* @author Martin Zvarík | zvarik.cz
* @version 2020-01-26
*/
class CPOSTOLD
{
	private $certPath;
	private $certPassword;
	private $idContract;
	
	public $debug = false; // set to TRUE to show XML DATA
	
	
	/**
	* __construct
	*
	* PFX certifikát byste měli mít na USB flash kartě.
	*
	* Případně je možno PFX certifikát získat z počítače, kde byl certifikát už nainstalován.
	* 1) Spustíme Internet Explorer -> Nastavení -> Možnosti internetu -> Obsah -> Certifikáty
	* 2) Vybereme certifikát -> Exportovat -> včetně Ano, exportovat privátní klíč (pokud toto máte zašedlé, jedná se o neúplnou instalaci certifikátu)
	* 3) Zvolíme typ PKCS č. 12 (PFX) -> Nastavíme heslo
	* 4) Uložíme na disk a ten soubor použijeme do této třídy
	*
	* @param (int) $id_contract - Např. 399998001 - Číslo je uvedeno ve smlouvě k B2B API
	* @param (string) $cert_file - Cesta k souboru PFX (certifikát), který jste dostali při registraci do B2B CPOST -- ve stejné složce musí být uveden i soubor "ca5_extra.pem"
	* @param (string) $cert_password - Heslo k certifikátu
	* @param (bool) $use_ca5 = true - Starší certifikáty byly vydávány pod autoritou CA3 a bylo třeba jim dát jiný podpis
	*/
	function __construct($id_contract, $cert_file, $cert_password, $use_ca5 = true)
	{
		$this->idContract = $id_contract;
		
		
		//------------- KONVERZE CERTIFIKATU
		
		$this->certPassword = $cert_password;
		
		$certPath = realpath(dirname($cert_file)); // potrebujeme uplnou cestu do CURL
		
		if (!file_exists($certPath.'/temp.cer') || !file_exists($certPath.'/temp.pem') || filemtime($cert_file) > filemtime($certPath.'/temp.cer'))
		{
			if (!file_exists($cert_file)) {
				throw new Exception('Certifikat "'.$cert_file.'" PFX nebyl nalezen.');
			}
			
			$pkcs12 = file_get_contents($cert_file);
			
			if (!extension_loaded('openssl') || !function_exists('openssl_pkcs12_read')) {
				throw new Exception('Rozsireni OpenSSL neni na serveru dostupne. Certifikaty (PFX => CER a PEM) muzete alternativne vygenerovat na mistnim PC pres prikazovou radku s aplikaci "openssl"');
				// openssl ... na .CER
				// openssl pkcs12 -in cert.pfx -out gen.pem -nodes
			}
			
			$res = [];
			$openSSL = openssl_pkcs12_read($pkcs12, $res, $cert_password);
			if(!$openSSL) {
				throw new Exception("Certifikat .CER se nepodarilo vyexportovat. ".openssl_error_string());
			}
			
			// 1) certifikát
			file_put_contents($certPath.'/temp.cer', $res['pkey'].$res['cert'].implode('', $res['extracerts']));
			
			if ($use_ca5) {
				// CA 5 certifikát autorit
				# Soubor byl vygenerován v tomto pořadí:
				# http://www.postsignum.cz/files/ca/postsignum_qca4_root.pem
				# http://www.postsignum.cz/files/ca/postsignum_qca4_sub.pem
				# QCA5
				# http://www.postsignum.cz/files/ca/postsignum_vca4_sub.pem
				# VCA5
				$sig = file_get_contents($certPath.'/ca5_extra.pem');
				$cert = $res['cert'].$sig;
			}
			else {
				$cert = $res['cert'].implode('', $res['extracerts']);
			}
			
			// 2) podpis
			file_put_contents($certPath.'/temp.pem', $cert);
		}
		
		$this->certPath = $certPath;
	}
	
		
	
	/**
	* Clean phone number, returns like +420732123456
	* Zformátuje telefon do čistého formátu
	*
	* @param (string) $s Phone Number
	* @param (string = CZ) $phonePrefix Country ISO code /or Prefix like +420 (will be added to $s only if needed)
	*/
	static function formatPhoneNumber($s, $phonePrefix = 'CZ')
	{
		$s = preg_replace("~(^00|(?<!^)\+|[^0-9\+])~", "", $s);
		
		if (strpos($s, '+') === FALSE)
		{
			$prefixesByIso = array(
				'CZ' => '+420',
				'SK' => '+421',
				'PL' => '+48',
				'DE' => '+49',
				'AT' => '+43',
				'CH' => '+41'
			);
			
			if (isset($phonePrefix) && isset($prefixesByIso[strtoupper($phonePrefix)])) {
				$phonePrefix = $prefixesByIso[strtoupper($phonePrefix)];
			}
			
			$noSignsPrefix = preg_replace("~[^0-9]~", "", $phonePrefix);
			if ($noSignsPrefix == substr($s, 0, strlen($noSignsPrefix))) {
				$s = '+' . $s; // only plus was missing
			}
			else {
				$s = $phonePrefix . $s;
			}
		}
		return $s;
	}
	
	
	/**
	* Délka řetězce
	*/
	private function getLength($str) {
		if (function_exists('mb_strlen')) {
			return mb_strlen($str, '8bit');
		} else {
			return strlen($str);
		}
	}
	
	
	/**
	* Pošle požadavek
	*
	* @param (string) $url - Lze i bez domény částečná
	* @param (string) $xmlData
	*/
	private function sendRequest($url, $xmlData)
	{
		if (strpos($url, 'https') === false) {
			$url = 'https://b2b.postaonline.cz/'.trim($url, '/');
		}
		
        $ch = curl_init();
		
		curl_setopt($ch, CURLOPT_URL, $url);
		
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'POST');
        
		curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 15);
		curl_setopt($ch, CURLOPT_TIMEOUT, 30);

		curl_setopt($ch, CURLOPT_SSLCERT, $this->certPath.'/temp.cer');
		curl_setopt($ch, CURLOPT_SSLCERTPASSWD, $this->certPassword);
		curl_setopt($ch, CURLOPT_CAINFO, $this->certPath.'/temp.pem');
		
		curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
//		curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
		curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
//		curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, CORE_TEST_MODE ? 0 : 1);
		// pro ssl na produkci ma byt 1, ale nefunguje

		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		
		$headers = array();
		
		// https://github.com/nategood/httpful/issues/37
        // Except header removes any HTTP 1.1 Continue from response headers
        $headers[] = 'Expect:';
		
		$headers[] = 'User-Agent: CeskaPosta/1.0 (cURL/7.61.1 PHP/7.2.13 (WINNT) Apache/2.4.7 (Win32) .2.13 Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/78.0.3904.108 Safari/537.36)';
		
		$headers[] = 'Content-Type: application/xml';
		
		$headers[] = 'Accept: */*; q=0.5, text/plain; q=0.8, text/html;level=3;';
		
		
		$t = explode(" ",microtime());
		$transactionId = date("YmdHis", $t[1]).substr((string)$t[0],1,4).rand(10,99); // interní číslo transakce
		
		$xmlData = '<?xml version="1.0" encoding="UTF-8" standalone="yes"?>
			<b2bRequest xmlns="https://b2b.postaonline.cz/schema/B2BCommon-v1" xmlns:ns2="https://b2b.postaonline.cz/schema/POLServices-v1">
				<header>
					<idExtTransaction>'.$transactionId.'</idExtTransaction>
					<timeStamp>'.date('c').'</timeStamp>
					<idContract>'.$this->idContract.'</idContract>
				</header>
				'.$xmlData.'
			</b2bRequest>';
		
		
		if ($this->debug || 0) {
			echo $xmlData; // DEBUG
			exit;
		}
		
		
		curl_setopt($ch, CURLOPT_POSTFIELDS, $xmlData);
		
		$headers[] = 'Content-Length: '.$this->getLength($xmlData);
		
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
		
		curl_setopt($ch, CURLOPT_VERBOSE, false);
		
        curl_setopt($ch, CURLOPT_HEADER, 0);
		
		$response = curl_exec($ch);
		
		if (empty($response)) {
			throw new Exception('CeskaPosta: Empty response. '.curl_error($ch));
		}
		
		$statusCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
		if ($statusCode != 200) {
			throw new Exception('CeskaPosta: Status Code = '.$statusCode);
		}
		
		$response = preg_replace('~(</?)([a-z0-9_\-]+):~i', '\1', $response);
		
		$xmlResponse = @simplexml_load_string($response);
		
		if (empty($xmlResponse)) {
			return $response;
		}
		
		if (isset($xmlResponse->serviceData)) {
			$xmlResponse = $xmlResponse->serviceData;
		}
		
		$k = substr(strrchr($url, '/'), 1);
		if (isset($xmlResponse->{$k."Response"})) {
			$xmlResponse = $xmlResponse->{$k."Response"};
		}
		
		return $xmlResponse;
	}
	
	
	/**
	* ARRAY to XML (simple function)
	*/
	private function arrayToXml($array, $prefix = '', $tabs = 0, $wraptags = '')
	{
		$ind = str_repeat("\t", $tabs);
		
		$out = $ind;
		
		foreach ($array as $k => $v)
		{
			if (is_array($v)) {
				if (is_int(key($v))) {
					// repeated block (n-entries)
					$out.= PHP_EOL . $this->arrayToXml($v, $prefix, $tabs+1, $wraptags.'<'.$prefix.$k.'>');
					continue;
				}
				
				$block = PHP_EOL . $this->arrayToXml($v, $prefix, $tabs+1);
			} else {
				$block = $v;
			}
			
			if (trim($block) == '') continue; // skip empty tags
			
			if (!empty($wraptags)) {
				$out.= $wraptags;
			}
			if (!is_int($k)) $out.= '<'.$prefix.$k.'>';
			$out.= $block;
			if (!is_int($k)) $out.= '</'.$prefix.$k.'>'.PHP_EOL.$ind;
			if (!empty($wraptags)) {
				$out.= str_replace('<', '</', $wraptags).PHP_EOL.$ind;
			}
		}
		
		if ($tabs > 0) {
			$out = substr($out, 0, -1); // smažeme poslední tab, protože konec sekce
		}
		
		return $out;
	}
	
	
	
	/**
	* Asynchronní zpracování
	* sendParcels - uloží data k zásilkám pro zpracování
	*
	* Operace slouží k předání dat k zásilkám do dalšího zpracování.
	* V případě, že zpracování proběhlo úspěšně, jsou data automaticky předána na podací poštu.
	* (Takto přijatá data k zásilkám je možno zobrazit prostřednictvím web aplikace POL.)
	*
	* V případě, že při zpracování byla zaznamenána chyba v datech, není do dalšího zpracování přijata ani jedna zásilka.
	* Výstupem této operace je element b2bASyncResponse, který obsahuje jednoznačné ID (idTransaction) potřebné pro zjištění výsledku zpracování operací getResultParcels.
	*
	* Pozn. 
	* - V případě, že je podací místo identifikováno prostřednictvím locationNumber, není nutná identifikace odesílatele prostřednictvím doParcelHeader.senderAddress 
	* - Do ukončení zpracování předaných dat vrací operace getResultParcels chybový kód 10 UNFINISHED_PROCESS - Zpracování není ještě ukončeno
	*/
	function sendParcels($data)
	{
		$data = '<serviceData>'.$this->arrayToXml(array('sendParcels' => $data), 'ns2:').'</serviceData>';
		
		return $this->sendRequest('services/POLService/v1/sendParcels', $data);
	}

	function sendParcelsSync($data)
	{
		$data = '<serviceData>'.$this->arrayToXml(array('parcelServiceSyncRequest' => $data), 'ns2:').'</serviceData>';

		return $this->sendRequest('services/POLService/v1/parcelServiceSync', $data);
	}
	
	
	/**
	* getResultParcels - vrací výsledky zpracování dat k podaným zásilkám
	*
	* @param (int) $idTransation - ID vrácené funkcí sendParcels
	*
	* @NOTE - Nelze volat více transakcí najednou!
	*/
	function getResultParcels($idTransaction)
	{
		$data = $this->arrayToXml(array('idTransaction' => $idTransaction));
		
		$response = $this->sendRequest('services/POLService/v1/getResultParcels', $data);
		
		
		/*
		CHYBY:
		- INFO_ADDRESS_WAS_MODIFIED //  scházející část obce
		- INVALID_PREFIX - Klient s technickým číslem nemá nasmlouvanou podací poštu (postCode)
		- MISSING_REQUIRED_SERVICE_4/5/41 - u dobírky je nutno uvést typ doplňkové služby
		- MISSING_VS - Chybí variabilní symbol
		- INFO_INEXACT_ADDRESS - Nepřesná adresa
		*/
		
		
		/*if (isset($response->doParcelHeaderResult)) {
			$response = $response->doParcelHeaderResult;
			
			$x = (array) $response;
			if ($x['doParcelStateResponse']) {
				$response = $x['doParcelStateResponse'];
			}
		}*/
		
		return $response;
	}
	
	
	
	/**
	* getStats - vrátí statistické informace o podáních za období
	*
	* @param (string/int) $dateBegin - date or timestamp
	* @param (string/int) $dateEnd - date or timestamp
	*/
	function getStats($dateBegin, $dateEnd)
	{
		if (!is_numeric($dateBegin)) $dateBegin = strtotime($dateBegin);
		if (!is_numeric($dateEnd)) $dateEnd = strtotime($dateEnd);
		
		if (!$dateEnd || !$dateBegin) throw new Exception('CPOST: Chyba datumu');
		
		
		$data = array(
			'serviceData' => array(
				'ns2:getStats' => array(
					'ns2:dateBegin' => date("c", $dateBegin),
					'ns2:dateEnd' => date("c", $dateEnd)
				)
			)
		);
		
		$data = $this->arrayToXml($data);
		
		$response = $this->sendRequest('services/POLService/v1/getStats', $data);
		
		return $response;
	}
	
	
	/**
	* getParcelState – vrátí stavy zásilek
	*
	* @param $idParcels (string/array) - 1 až 10 čísel
	* @param $language $str - Jazyk v jakém budou vráceny výsledky
	*/
	function getParcelState($idParcels, $language = 'cs')
	{
		$data = array('serviceData' => array(
			'ns2:getParcelState' => array(
				'ns2:idParcel' => $idParcels,
				'ns2:language' => strtolower($language)
			)
		));
		
		$data = $this->arrayToXml($data);
		
		$response = $this->sendRequest('services/POLService/v1/getParcelState', $data);
		
		$x = (array) $response;
		if ($x['parcel']) {
			$response = $x['parcel'];
		}
		
		return $response;
	}
	
	
	/**
	* getParcelsPrinting - vrácí adresní štítky 
	*/
	function getParcelsPrinting($data)
	{
		$data = '<serviceData>'.$this->arrayToXml(array('getParcelsPrinting' => $data), 'ns2:').'</serviceData>';
		
		$response = $this->sendRequest('services/POLService/v1/getParcelsPrinting', $data);
		
		return $response;
	}
	
	
	/**
	* ConsignmentStatuses – vrací data zásilek pro zadaný časový úsek a technologické číslo podavatele (customerID)
	*
	* @param (string/int) $dateBegin - date or timestamp
	* @param (string/int) $dateEnd - date or timestamp
	* @param (string) $customerId Technologické číslo např. U516
	*
	*
	* @document [WS Zásilka] Druhý word dokument
	*/
	function ConsignmentStatuses($dateBegin, $dateEnd, $customerId)
	{
		if (!is_numeric($dateBegin)) $dateBegin = strtotime($dateBegin);
		if (!is_numeric($dateEnd)) $dateEnd = strtotime($dateEnd);
		
		if (!$dateEnd || !$dateBegin) throw new Exception('CPOST: Chyba datumu');
		
		
		$data = array(
			'serviceData' => array(
				'ns2:consignmentRequest' => array(
					'ns2:timeStampFrom' => date("c", $dateBegin),
					'ns2:timeStampTo' => date("c", $dateEnd),
					'ns2:customerID' => $customerId,
				)
			)
		);
		
		$data = $this->arrayToXml($data);
		
		$response = $this->sendRequest('services/ConsignmentServices/v1/ConsignmentStatuses', $data);
		
		return $response;
	}
	
	
}


