<?php declare(strict_types = 1);

namespace Ceskaposta\Model\Libs;

use Ceskaposta\Model\AdvancedOptions;
use Ceskaposta\Model\CeskaPostaConfig;
use Ceskaposta\Model\Classes\AsyncParcel;
use Ceskaposta\Model\Classes\SyncParcel;
use Ceskaposta\Model\Entities\ICeskaPostaNumberPackage;
use Ceskaposta\Model\Entities\ICeskaPostaOrder;
use Ceskaposta\Model\Entities\ParcelDeliveryToHandOrder;
use Ceskaposta\Model\Entities\PostOfficeOrder;
use Ceskaposta\Model\Entities\PostWarehouseOrder;
use Ceskaposta\Model\Events\SendOrdersEvent;
use Ceskaposta\Model\Helper;
use Contributte\Translation\Exceptions\InvalidArgument;
use Contributte\Translation\Translator;
use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\Strings;
use EshopOrders\Model\ExpeditionLogger;
use EshopOrders\Model\Helpers\OrderHelper;
use Exception;
use Nette\Utils\DateTime;
use Nette\Utils\FileSystem;
use Nettrine\ORM\EntityManagerDecorator;
use SimpleXMLElement;
use Tracy\Debugger;

class CeskaPostaApi
{
	public const LIMIT_LABELS = 500;
	public const LABELS_DIR   = TMP_DIR . '/eshopOrders/ceskapostaLabels/';

	protected EventDispatcher        $eventDispatcher;
	protected array                  $config;
	protected EntityManagerDecorator $em;
	protected Translator             $translator;
	protected ?CeskaPostaApiBridge   $cpostApi = null;
	protected ExpeditionLogger       $expeditionLogger;

	public function __construct(
		EventDispatcher        $eventDispatcher,
		EntityManagerDecorator $em,
		Translator             $translator,
		CeskaPostaApiBridge    $cpostApiBridge,
		ExpeditionLogger       $expeditionLogger
	)
	{
		$this->eventDispatcher  = $eventDispatcher;
		$this->em               = $em;
		$this->translator       = $translator;
		$this->expeditionLogger = $expeditionLogger;

		if (CeskaPostaConfig::load('enable')) {
			$this->cpostApi = $cpostApiBridge;
		}
	}

	/**
	 * @param ICeskaPostaOrder[] $orders
	 *
	 * @return array<int, ICeskaPostaOrder[]>
	 * @throws Exception
	 */
	protected function sortOrdersByTransmissionDate(array $orders): array
	{
		$transmissionDateFromConfig = $this->cpostApi->getConfig()['transmissionDate'];
		$key                        = $transmissionDateFromConfig instanceof DateTime ? $transmissionDateFromConfig : new DateTime($transmissionDateFromConfig);

		$result = [];

		foreach ($orders as $order) {
			$result[$key->getTimestamp()][] = $order;
		}

		return $result;
	}

	protected function getParcelData(ICeskaPostaOrder $order, int $quantityParcel = 1, int $sequenceParcel = 1, AdvancedOptions $advancedOptions): array
	{
		/** @var AsyncParcel|SyncParcel $parcel */
		$parcel          = $quantityParcel > 1 ? AsyncParcel::class : SyncParcel::class;
		$o               = $order->getOrder();
		$price           = strtolower($o->getCurrencyCode()) === 'czk' ? ceil($o->getPrice()) : $o->getPrice();
		$customer        = $o->getCustomer();
		$addressDelivery = $o->getAddressDelivery();
		$isCod           = $o->getPaymentIdent() === 'cod';

		if ($isCod) {
			$curr  = $o->getCurrencyCode();
			$price = round($price, $curr === 'CZK' ? 0 : $o->currency->decimals);

			if (
				(!$addressDelivery && $curr === 'EUR')
				|| ($addressDelivery && Strings::lower((string) $addressDelivery->getCountry()->getId()) === 'sk')
			) {
				$price = OrderHelper::roundSkCod($price);
			}
		}

		$data = [
			$parcel::PARAMS => [
				'recordID'         => $o->getId(),
				'parcelCode'       => '',
				'prefixParcelCode' => $advancedOptions->getUnifiedPrefix(),
				'weight'           => $advancedOptions->getWeight(),
				'insuredValue'     => $advancedOptions->getInsuredValue(),
				'amount'           => $isCod ? $price : 0,
				'currency'         => strtoupper($o->getCurrencyCode()),
				'vsVoucher'        => $advancedOptions->getVariableSymbol(),
				'vsParcel'         => $advancedOptions->getVariableSymbol(),
			],
		];

		if ($advancedOptions->getExportNote()) {
			$data[$parcel::PARAMS]['notePrint'] = $advancedOptions->getExportNote();
		}

		if ($quantityParcel > 1) {
			$data[$parcel::PARAMS]['quantityParcel'] = $quantityParcel;
			$data[$parcel::PARAMS]['sequenceParcel'] = $sequenceParcel;

			// u druheho a dalsiho kusu netreba uvadet udanou cenu a dobirku
			if ($sequenceParcel > 1) {
				unset($data[$parcel::PARAMS]['insuredValue']);
				unset($data[$parcel::PARAMS]['amount']);
			}
		}

		if ($customer) {
			$data[$parcel::ADDRESS] = [
				'recordID' => $customer->getId(),
			];
		}

		foreach ([
			         'firstName'    => 'firstName',
			         'surname'      => 'lastName',
			         'companyName'  => 'company',
			         'street'       => 'street',
			         'city'         => 'city',
			         'zipCode'      => 'postal',
			         'mobileNumber' => 'phone',
			         'emailAddress' => 'email',
		         ] as $k => $v) {
			$methodName = sprintf('get%s', ucfirst($v));
			if (method_exists($addressDelivery, $methodName) && !empty($addressDelivery->$methodName())) {
				if ($k === 'mobileNumber') {
					$data[$parcel::ADDRESS][$k] = self::decoratePhone($addressDelivery->$methodName());
				} else {
					$data[$parcel::ADDRESS][$k] = self::truncateParcelAddress($k, $addressDelivery->$methodName());
				}
			}
		}

		if ($order instanceof PostOfficeOrder || $order instanceof PostWarehouseOrder) {
			$data[$parcel::ADDRESS]['street']  = '';
			$data[$parcel::ADDRESS]['city']    = self::truncateParcelAddress('city', $order->getPostName() ?? '');
			$data[$parcel::ADDRESS]['zipCode'] = self::truncateParcelAddress('zipCode', $order->getPostPSC());
		}

		if ($addressDelivery->getCountry() !== null) {
			$data[$parcel::ADDRESS]['isoCountry'] = $addressDelivery->getCountry()->getId();
		}

		$data[$parcel::ADDRESS]['houseNumber'] = '';

		/**
		 * SLUZBY
		 * 7 - udana cena
		 * 41 - bezdokladova dobirka
		 * 46 - email avizo
		 * 34 - SMS
		 * 70 - kdyz je zasilka vicekusova
		 * ...
		 */
		foreach ($advancedOptions->getServices($quantityParcel) as $s) {
			$arr = [
				'service' => $s,
			];
			if ($quantityParcel > 1) {
				$data[$parcel::SERVICES][] = $arr;
			} else {
				$data[$parcel::PARAMS][$parcel::SERVICES][] = $arr;
			}
		}

		return $data;
	}

	/**
	 * @param ICeskaPostaOrder[] $orders
	 *
	 * @throws Exception
	 */
	public function generateLabels(array $orders): array
	{
		if (!$orders) {
			return [];
		}
		$ordersCount = count($orders);
		if ($ordersCount > self::LIMIT_LABELS) {
			Debugger::log(sprintf('CP generateLabels(): $orders must have from 1 to %s items', self::LIMIT_LABELS), 'cpost');
		}

		$orders  = array_slice($orders, 0, self::LIMIT_LABELS);
		$numbers = array_map(static fn(ICeskaPostaOrder $order) => $order->getNumberPackage(), $orders);

		foreach ($orders as $o) {
			foreach ($o->getAssociatedNumberPackages() as $anp) {
				$numbers[] = $anp->getNumberPackage();
			}
		}
		$numbers = array_values(array_unique($numbers));
		$result  = [
			'ok'    => 0,
			'error' => 0,
			'files' => [],
		];

		if ($this->cpostApi === null) {
			return $result;
		}

		$config = $this->cpostApi->getConfig();
		$data   = [
			'doPrintingHeader' => [
				'customerID'      => $config['customerId'],
				'idForm'          => $config['label']['idForm'],
				'shiftHorizontal' => $config['label']['shiftHorizontal'],
				'shiftVertical'   => $config['label']['shiftVertical'],
				'position'        => $config['label']['position'],
			],
			'doPrintingData'   => [
				'parcelCode' => $numbers,
			],
		];

		$response = $this->cpostApi->getParcelsPrinting($data);

		if ($response['file']) {
			$file = self::LABELS_DIR . time() . '_' . uniqid() . '.pdf';
			FileSystem::createDir(dirname($file));
			$pdf = fopen($file, 'w');
			fwrite($pdf, base64_decode($response['file']));
			fclose($pdf);
			$result['files'][] = $file;
			$result['ok']++;
		} else {
			//			Debugger::log(((array) $response->doPrintingHeaderResult->doPrintingStateResponse)['responseText'], 'ceskaposta-api-label');
			Debugger::log(['response' => $response, 'data' => $data], 'ceskaposta-api-label');
			if (isset($response['data']['printingStatusResponse'])) {
				$errCode = $response['data']['printingStatusResponse']['responseCode'];
				$respText = $response['data']['printingStatusResponse']['responseText'];
				$err = sprintf('%s - %s', $errCode, $respText);

				foreach ($orders as $o) {
					$this->expeditionLogger->logError('cpost', $err, $o->getOrder()->getId());
				}
			}
			$result['error']++;
		}

		// Odstraneni starych stitku
		$today = (new DateTime())->format('Y-m-d');
		foreach (glob(self::LABELS_DIR . '*.pdf') as $file) {
			$time = explode('_', basename($file))[0];

			if (Strings::isValidTimestamp((int) $time)) {
				$date = DateTime::from($time)->format('Y-m-d');

				if ($date < $today) {
					unlink($file);
				}
			}
		}

		return $result;
	}

	/**
	 * @param ICeskaPostaOrder[] $orders
	 *
	 * @throws Exception
	 */
	public function sendOrders(array $orders, int $quantity = 1, array $advancedOptions = []): array
	{
		$config = $this->cpostApi->getConfig();
		$parcels      = [];
		$orderIds     = [];
		$result       = [
			'ok'           => 0,
			'error'        => 0,
			'message'      => '',
			'messageClass' => '',
		];
		$ordersSorted = $this->sortOrdersByTransmissionDate($orders);

		/** @var AsyncParcel|SyncParcel $parcel */
		$parcel = $quantity > 1 ? AsyncParcel::class : SyncParcel::class;

		$event = new SendOrdersEvent($ordersSorted);
		$this->eventDispatcher->dispatch($event, 'ceskaposta.beforeSendOrders');
		$ordersSorted = $event->orders;

		if (!$ordersSorted || $this->cpostApi === null) {
			return $result;
		}

		$tmp = $ordersSorted[array_key_first($ordersSorted)];

		/** @var ICeskaPostaOrder $currOrder */
		$currOrder = $tmp[array_key_first($tmp)];

		// pokud byla zasilka již drive odeslana (drive mohla skoncit stavem UNFINISHED_PROCESS), pak pouziji existujici $idTransaction
		if (($idTransaction = $currOrder->getIdTransaction()) === null) {
			/**
			 * @var int                $timestamp
			 * @var ICeskaPostaOrder[] $ordersArray
			 */
			foreach ($ordersSorted as $timestamp => $ordersArray) {
				$count = count($ordersArray);
				if ($count === 0) {
					continue;
				}
				$parcels = [
					$parcel::HEADER => [
						'transmissionDate' => DateTime::from($timestamp)->format('d.m.Y'),
						'customerID'       => $config['customerId'],
						'postCode'         => $config['postCode'],
						'locationNumber'   => $config['locationNumber'],
					],
				];

				if ($count === 1) {
					$o    = $ordersArray[array_key_first($ordersArray)];
					$oId  = $o->getOrder()->getId();
					$adOp = new AdvancedOptions($o, $advancedOptions);
					if ($quantity > 1) {
						for ($i = 1; $i <= $quantity; $i++) {
							$parcels[$parcel::DATA][] = $this->getParcelData($o, $quantity, $i, $adOp);
						}
					} else {
						$parcels[$parcel::DATA] = $this->getParcelData($o, 1, 1, $adOp);
					}
					$orderIds[$oId] = $oId;
				} else {
					$i = 1;
					foreach ($ordersArray as $order) {
						$oId  = $order->getOrder()->getId();
						$adOp = new AdvancedOptions($order, $advancedOptions);

						// vicekus nejde zasilat na balikovnu
						if ($quantity > 1 && Strings::upper($order->getPrefixParcelCode()) !== PostWarehouseOrder::PREFIX_PARCEL_CODE) {
							for ($i = 1; $i <= $quantity; $i++) {
								$parcels[$parcel::DATA][] = $this->getParcelData($order, $quantity, $i, $adOp);
							}
						} else {
							$parcels[$parcel::DATA] = $this->getParcelData($order, 1, 1, $adOp);
						}
						$orderIds[$oId] = $oId;
						$i++;
					}
				}
			}

			if ($quantity > 1) {
				$idTransaction = $this->cpostApi->sendParcels($parcels); // pri synchronni odeslani to hlasi chybu v datech, proto async
				$result['error'] = count($orderIds);
				if (!$idTransaction) {
					return $result;
				}
			} else {
				$response        = $this->cpostApi->sendParcelsSync($parcels);
				$result['error'] = count($orderIds);
			}
		}

		$numberPackages = [];
		if ($idTransaction !== null) { // zpracovani async
			$parcelNumbers = $this->cpostApi->getResultParcels($idTransaction);

			// pokud je vracen UNFINISHED_PROCESS, zeptame se jeste jednou
			if (isset($parcelNumbers['errorCode']) && $parcelNumbers['errorCode'] === '10') {
				$parcelNumbers = $this->cpostApi->getResultParcels($idTransaction);
			}

			$responseCode = $parcelNumbers['errorCode'];
			if (!$parcelNumbers['parcels'] || $responseCode === '19') { // 19 je BATCH_INVALID - krome warningu jsou v response i chyby
				$entity = $currOrder;
				$arr    = (array) $idTransaction;
				$entity->setIdTransaction($arr[array_key_first($arr)]);
				$this->em->persist($entity);
				$this->em->flush($entity);
				if ($responseCode !== '19') {
					$result['message'] = $this->translator->translate('ceskaposta.export.unfinishedProcess', ['order' => $entity->getOrder()->getId()]);
					/** @phpstan-ignore-next-line */
				} else if (isset($response)) {
					/** @phpstan-ignore-next-line */
					$result['message'] = $this->processErrors($response['responseHeader']['resultParcelData']['parcelDataResponse']);
				}
				$result['messageClass'] = 'danger';

				return $result;
			}

			$numberPackages = [];
			foreach ($parcelNumbers['parcels'] as $r) {
				$parcelCode = str_replace(' ', '', (string) $r['parcelCode']);
				$orderId    = (int) $r['recordNumber'];

				if (!$parcelCode) {
					continue;
				}

				$numberPackages[$orderId][$parcelCode] = $parcelCode;
			}
		} else if (isset($response)) { // zpracovani sync
			$responseHeader = $response['responseHeader']['resultHeader'];
			$responseCode   = (int) $responseHeader['responseCode'];
			if ($responseCode !== 1) {
				$result['message']      = $this->processErrors($response['responseHeader']['resultParcelData']['parcelDataResponse']);
				$result['messageClass'] = 'danger';

				return $result;
			}

			foreach ($response['responseHeader']['resultParcelData'] as $r) {
				$parcelCode = (string) $r['parcelCode'];
				$orderId    = (int) $r['recordNumber'];

				if (!$parcelCode) {
					continue;
				}

				$numberPackages[$orderId][$parcelCode] = $parcelCode;
			}
		}

		$ordersById = [];
		foreach ($orders as $order) {
			$ordersById[$order->getOrder()->getId()] = $order;
		}

		foreach ($numberPackages as $orderId => $parcelCodes) {
			$i = 0;
			foreach ($parcelCodes as $parcelCode) {
				if ($i === 0) {
					// ulozeni parcel code do db k objednavce
					foreach (Helper::getClassesByParcelCode($parcelCode) as $class) {
						/** @var PostOfficeOrder|PostWarehouseOrder|ParcelDeliveryToHandOrder|null $entity */
						/** @phpstan-ignore-next-line */
						$entity = $this->em->getRepository($class)->find($orderId);
						if (!$entity) {
							continue;
						}

						$entity->numberPackage = $parcelCode;
						$entity->idTransaction = null;
						$entity->export();
						$this->em->persist($entity);
						break;
					}

				} else if (isset($entity)) {
					$class = Helper::getAssociatedClassByMainClass(get_class($entity));
					/** @var ICeskaPostaNumberPackage $en */
					$en = new $class($parcelCode, $entity);
					$this->em->persist($en);
				}
				$i++;
			}

			if (isset($ordersById[$orderId]) && isset($entity)) {
				$spedition = $ordersById[$orderId]->getOrder()->getSpedition();

				$spedition->trackingNumber = $entity->getNumberPackage();
				$spedition->trackingUrl    = $entity->getTrackingUrl();
				$this->em->persist($spedition);
			}

			unset($orderIds[$orderId]);
			$result['ok']++;
		}
		$this->em->flush();

		$result['error'] = count($orderIds);

		return $result;
	}

	public function checkCompleted(ICeskaPostaOrder $order, string $packageNumber): bool
	{
		if ($this->cpostApi === null) {
			return false;
		}
		$states = $this->cpostApi->getParcelState($packageNumber);

		foreach ($states as $state) {
			if (((int) $state['id']) === 91) { // 91 - dodani zasilky
				return true;
			}
		}

		return false;
	}

	/**
	 * @param string $val
	 */
	protected static function truncateParcelAddress(string $k, $val): string
	{
		$val = str_replace('&&', '&', $val);
		$val = str_replace('&', ' a ', $val);
		$val = str_replace('  ', ' ', $val);
		switch ($k) {
			case 'firstName':
			case 'surname':
			case 'companyName':
				return Strings::substring($val, 0, 50);
			case 'street':
			case 'city':
				return Strings::substring($val, 0, 40);
			case 'zipCode':
				return substr($val, 0, 25);
			default:
				return $val;
		}
	}

	protected static function decoratePhone(string $phone): string
	{
		$s           = preg_replace("~(^00|(?<!^)\+|[^0-9\+])~", "", $phone);
		$phonePrefix = 'CZ';

		if (Strings::startsWith($s, '421')) {
			$phonePrefix = 'SK';
		} else if (Strings::startsWith($s, '48')) {
			$phonePrefix = 'PL';
		} else if (Strings::startsWith($s, '49')) {
			$phonePrefix = 'DE';
		} else if (Strings::startsWith($s, '43')) {
			$phonePrefix = 'AT';
		} else if (Strings::startsWith($s, '41')) {
			$phonePrefix = 'CH';
		}

		return CPOSTOLD::formatPhoneNumber($phone, $phonePrefix);
	}

	/**
	 * @param SimpleXMLElement|array $parcelDataResponse
	 *
	 * @throws InvalidArgument
	 */
	protected function processErrors($parcelDataResponse): string
	{
		if ($parcelDataResponse instanceof SimpleXMLElement) {
			return $this->translator->translate(sprintf('ceskaposta.export.errors.%s', $parcelDataResponse->responseText));
		}

		if (is_array($parcelDataResponse) && count($parcelDataResponse) > 0) {
			return implode(' ', array_map(fn($el) => sprintf('[%s]', $this->translator->translate(sprintf('ceskaposta.export.errors.%s', $el->responseText))), $parcelDataResponse));
		}

		return $this->translator->translate('ceskaposta.export.globalError');
	}

}
