<?php declare(strict_types = 1);

namespace Ceskaposta\Model;

use Ceskaposta\Model\Entities\ICeskaPostaOrder;
use Ceskaposta\Model\Entities\ParcelDeliveryToHandOrder;
use Ceskaposta\Model\Entities\PostOfficeOrder;
use Ceskaposta\Model\Entities\PostWarehouseOrder;
use Doctrine\ORM\EntityRepository;
use Doctrine\ORM\Query\Expr\Join;
use EshopOrders\Model\Entities\OrderStatus;
use Nettrine\ORM\EntityManagerDecorator;

class OrdersExported
{
	protected PostOfficesOrder           $postOfficesOrder;
	protected PostWarehousesOrder        $postWarehousesOrder;
	protected ParcelsDeliveryToHandOrder $parcelsDeliveryToHandOrder;
	protected EntityManagerDecorator     $em;

	public function __construct(PostOfficesOrder $postOfficesOrder, PostWarehousesOrder $postWarehousesOrder, ParcelsDeliveryToHandOrder $parcelsDeliveryToHandOrder, EntityManagerDecorator $em)
	{
		$this->postOfficesOrder           = $postOfficesOrder;
		$this->postWarehousesOrder        = $postWarehousesOrder;
		$this->parcelsDeliveryToHandOrder = $parcelsDeliveryToHandOrder;
		$this->em                         = $em;
	}

	/**
	 * @param int[]|string[] $ids
	 * @return ICeskaPostaOrder[]
	 */
	public function getOrdersNotExported(array $ids = []): array
	{
		return $this->getOrders(false, $ids);
	}

	/**
	 * @param int[]|string[] $ids
	 * @return ICeskaPostaOrder[]
	 */
	public function getOrdersExported(array $ids = [], bool $loadFull = true): array
	{
		return $this->getOrders(true, $ids, $loadFull);
	}

	/**
	 * @param int[]|string[] $ids
	 * @return ICeskaPostaOrder[]
	 */
	public function getOrders(?bool $isExported = false, array $ids = [], bool $loadFull = true): array
	{
		$result = [];
		array_push($result, ...$this->getOrdersResult($this->postOfficesOrder->getEr(), $isExported, $ids, $loadFull));
		array_push($result, ...$this->getOrdersResult($this->postWarehousesOrder->getEr(), $isExported, $ids, $loadFull));
		array_push($result, ...$this->getOrdersResult($this->parcelsDeliveryToHandOrder->getEr(), $isExported, $ids, $loadFull));

		$r = [];
		foreach ($result as $o) {
			$r[$o->getOrder()->getId()] = $o;
		}

		return $r;
	}

	protected function getOrdersResult(EntityRepository $er, ?bool $isExported = false, array $ids = [], bool $loadFull = true): array
	{
		$qb = $er->createQueryBuilder('go')
			->leftJoin('go.associatedNumberPackages', 'anp')
			->orderBy('go.order')
			->groupBy('go.order');

		if ($loadFull) {
			$qb->addSelect('o, oad, oai, os, s, oCurrency')
				->innerJoin('go.order', 'o')
				->leftJoin('o.currency', 'oCurrency')
				->leftJoin('o.addressDelivery', 'oad')
				->leftJoin('o.addressInvoice', 'oai')
				->leftJoin('o.spedition', 'os')
				->leftJoin('os.spedition', 's');
		} else {
			$qb->addSelect('anp');
		}

		if ($isExported === true) {
			$qb->where('go.exported IS NOT NULL');
		} else if ($isExported === false) {
			$qb->where('go.exported IS NULL');
		}

		if (!empty($ids)) {
			$qb->andWhere('go.order IN (:ids)')->setParameter('ids', $ids);
		}

		$result = [];
		foreach ($qb->getQuery()->getResult() as $item) {
			$result[$item->getOrder()->getId()] = $item;
		}

		return $result;
	}

	/**
	 * @return ICeskaPostaOrder[]
	 */
	public function getOrdersNotCompleted(): array
	{
		$result = [];
		array_push($result, ...$this->getOrdersNotCompletedByEntity(PostOfficeOrder::class));
		array_push($result, ...$this->getOrdersNotCompletedByEntity(PostWarehouseOrder::class));
		array_push($result, ...$this->getOrdersNotCompletedByEntity(ParcelDeliveryToHandOrder::class));

		$r = [];
		foreach ($result as $o) {
			$r[$o->getOrder()->getId()] = $o;
		}

		return $r;
	}

	/**
	 * @return ICeskaPostaOrder[]
	 */
	protected function getOrdersNotCompletedByEntity(string $entity): array
	{
		$data = [];
		foreach ($this->em->createQueryBuilder()
			         ->select('o, zo, site, GROUP_CONCAT(s.id) as hidden statuses')
			         ->from($entity, 'zo')
			         ->join('zo.order', 'o')
			         ->join('o.orderStatuses', 'os', Join::WITH, 'os.deleted IS NULL OR os.deleted = 0')
			         ->join('os.status', 's')
			         ->join('o.site', 'site')
			         ->andWhere('zo.numberPackage IS NOT NULL')
			         ->andHaving('statuses LIKE :stat')
			         ->setParameters([
				         'stat' => '%' . OrderStatus::STATUS_SPEDITION,
			         ])
			         ->groupBy('o.id')
			         ->addOrderBy('o.id')
			         ->getQuery()->getResult() as $row) {
			$data[$row->getOrder()->getId()] = $row;
		}

		return $data;
	}

	public function findIdByPackageNumber(string $packageNumber): array
	{
		return $this->findIdByPackageNumberByEntity(PostOfficeOrder::class, $packageNumber)
			+ $this->findIdByPackageNumberByEntity(PostWarehouseOrder::class, $packageNumber)
			+ $this->findIdByPackageNumberByEntity(ParcelDeliveryToHandOrder::class, $packageNumber);
	}

	public function findIdByPackageNumberByEntity(string $entity, string $packageNumber): array
	{
		$ids = [];
		foreach ($this->em->createQueryBuilder()
			         ->select('IDENTITY(c.order) as id')
			         ->from($entity, 'c')
			         ->leftJoin('c.associatedNumberPackages', 'acp')
			         ->orWhere('c.numberPackage LIKE :number')
			         ->orWhere('acp.numberPackage LIKE :number')
			         ->setParameter('number', "%$packageNumber%")
			         ->getQuery()->getArrayResult() as $row) {
			$ids[] = $row['id'];
		}

		return $ids;
	}
}
